/**************************************************************************************
void Pnm/
Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.
***************************************************************************************

Description:
 PROFINET IO Controller V3 DPM API definition

 All references for profinet specification refer to the documents IEC
 specification documents IEC 61158610, IEC 61158510 and IEC 617842
 starting from version 2.3Ed2Mu3. These documented can be obtain from
 PNO.
**************************************************************************************/
/**
 * @file PNM_AP_Public.h
 *  PNM V3 DPM API definitions
 *
 */
#ifndef __PNM_AP_PUBLIC_H
#define __PNM_AP_PUBLIC_H

#include "stdint.h"

/* packing of structure members */

#if defined (__GNUC__)
#if !defined(__PACKED_PRE) || !defined(__PACKED_POST)
  #define __PACKED_PRE
  #define __PACKED_POST  __attribute__ ((packed))
#endif

#elif _MSC_VER >= 1000
#if !defined(__PACKED_PRE) || !defined(__PACKED_POST)
  #define __PACKED_PRE
  #define __PACKED_POST
#endif
  #pragma pack (1)
#endif

/** \def ARRAYS_OF_LENGTH_ZERO

  The macro ARRAYS_OF_LENGTH_ZERO can be used for array with variable-length object. Like:
  \code
  struct line {
    int length;
    char contents[0];
  };
  \endcode

  \note If the used Compiler is not be able to use ARRAYS_OF_LENGTH_ZERO with value 0 you have to define ARRAYS_OF_LENGTH_ZERO with e.g. 1
*/
#ifndef ARRAYS_OF_LENGTH_ZERO
  #define ARRAYS_OF_LENGTH_ZERO
#endif


/** @defgroup pnm_ap_dpmstructures
 *
 *  Additional dual port memory structures defined by the profinet IO
 *  controller
 */

typedef struct PNS_AP_TIMEINFO_Ttag PNS_AP_TIMEINFO_T;

/** A time point using profinet notation */
__PACKED_PRE struct __PACKED_POST PNS_AP_TIMEINFO_Ttag
{
  /** the cycle counter associated with the event */
  uint32_t ulCycleCounter;
  /** the time passed since cycle start associated with the event
   *  in nanoseconds */
  uint32_t ulTimeSinceCycleStart;
};


typedef struct PNM_AP_IOTIMINGINFO_Ttag PNM_AP_IOTIMINGINFO_T;

/** Timing information related to a certain state in process data
 *  processing.
 *
 *  This structure is used to provide the application with timing information
 *  at certain states in io data processing. It can be used to monitor
 *  the performance of the protocol stack
 */
__PACKED_PRE struct __PACKED_POST PNM_AP_IOTIMINGINFO_Ttag
{
  /** the time point when the application requested the firmware to take/home/andi/Projekte
   * over output data
   */
  PNS_AP_TIMEINFO_T tOutputUpdateStart;
  /** the time point when the firmware finished updating the output
   * data buffers
   *
   * \note This is not the time when the dpm output handshake is confirmed
   *       by the firmware
   */
  PNS_AP_TIMEINFO_T tOutputUpdateFinish;
  /** the time point when the firmware starts updating the dpm input area
   * from the input data buffsers.
   *
   * \note: This is not the time when the application triggers the dpm input
   *        area handshake. This is roughly the time when the red irt phase
   *        ends.
   */
  PNS_AP_TIMEINFO_T tInputUpdateStart;
  /** The time point when the firmware finished updating the dpm input area
   * from the input data buffers. At this time the dpm input area handshake
   * has been confirmed by the firmware. */
  PNS_AP_TIMEINFO_T tInputUpdateFinish;
};



/* allow compilation without including the whole TLR stuff */
#ifdef __TLR_TYPES_H
#define PNM_AP_PCK_HEADER_T TLR_PACKET_HEADER_T
#else
/** common packet header definitions
 *
 * This is just the standard PNM_AP_PCK_HEADER_T structure. As we want to avoid
 * any TLR/TLS based definitions from the beginning, I decided to redefine the structure
 * here in order to avoid including the TLR headers */
typedef struct PNM_AP_PCK_HEADER_Ttag PNM_AP_PCK_HEADER_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_PCK_HEADER_Ttag
{
  uint32_t  ulDest;   /* destination of the packet (task message queue reference) */
  uint32_t  ulSrc;    /* source of the packet (task message queue reference) */
  uint32_t  ulDestId; /* destination reference (internal use for message routing) */
  uint32_t  ulSrcId;  /* source reference (internal use for message routing) */
  uint32_t  ulLen;    /* length of packet data (starting from the end of the header) */
  uint32_t  ulId;     /* identification reference (internal use by the sender) */
  uint32_t  ulSta;    /* operation status code (error code, initialize with 0) */
  uint32_t  ulCmd;    /* operation command code */
  uint32_t  ulExt;    /* extension count (nonzero in multi-packet transfers) */
  uint32_t  ulRout;   /* router reference (internal use for message routing) */
} ;
#endif

/** empty packet without data part definition */
typedef struct PNM_AP_EMPTY_PCK_Ttag PNM_AP_EMTPY_PCK_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_EMPTY_PCK_Ttag
{
  PNM_AP_PCK_HEADER_T tHead;
};

/*************************************************************************************/


/** UUID Structure */
typedef struct PNM_UUID_Ttag PNM_UUID_T;
__PACKED_PRE struct __PACKED_POST PNM_UUID_Ttag
{
  uint32_t              ulUuidData1;
  uint16_t              usUuidData2;
  uint16_t              usUuidData3;
  uint8_t               abUuidData4[8];
} ;

#define PNM_AP_CFG_PACKET_COMMAND_START              = (0x00009400)

#define PNM_AP_MAX_ALARM_DATA_LENGTH                 (1024)
enum PNM_AP_CMD_Etag
{
  /* configuration services */
  PNM_AP_CMD_CFG_IOC_REQ                             = 0x00009400,
  PNM_AP_CMD_CFG_IOC_CNF                             = 0x00009401,
  PNM_AP_CMD_CFG_IOC_PRM_REQ                         = 0x00009402,
  PNM_AP_CMD_CFG_IOC_PRM_CNF                         = 0x00009403,
  PNM_AP_CMD_CFG_IOD_REQ                             = 0x00009404,
  PNM_AP_CMD_CFG_IOD_CNF                             = 0x00009405,
  PNM_AP_CMD_CFG_AR_PRM_REQ                          = 0x00009406,
  PNM_AP_CMD_CFG_AR_PRM_CNF                          = 0x00009407,
  PNM_AP_CMD_CFG_IOCR_REQ                            = 0x00009408,
  PNM_AP_CMD_CFG_IOCR_CNF                            = 0x00009409,
  PNM_AP_CMD_CFG_SUBMODULE_REQ                       = 0x0000940A,
  PNM_AP_CMD_CFG_SUBMODULE_CNF                       = 0x0000940B,
  PNM_AP_CMD_CFG_RECORD_REQ                          = 0x0000940C,
  PNM_AP_CMD_CFG_RECORD_CNF                          = 0x0000940D,
  PNM_AP_CMD_CFG_TOPO_REQ                            = 0x0000940E,
  PNM_AP_CMD_CFG_TOPO_CNF                            = 0x0000940F,
  PNM_AP_CMD_CFG_DOWNLOAD_FINISH_REQ                 = 0x00009410,
  PNM_AP_CMD_CFG_DOWNLOAD_FINISH_CNF                 = 0x00009411,
  PNM_AP_CMD_CFG_OEMPARAM_REQ                        = 0x00009412,
  PNM_AP_CMD_CFG_OEMPARAM_CNF                        = 0x00009413,

  /* device access ar */
  PNM_AP_CMD_ESTABLISH_DAAR_REQ                      = 0x0000941C,
  PNM_AP_CMD_ESTABLISH_DAAR_CNF                      = 0x0000941D,
  PNM_AP_CMD_RELEASE_DAAR_REQ                        = 0x0000941E,
  PNM_AP_CMD_RELEASE_DAAR_CNF                        = 0x0000941F,

  /* acyclic & ar services */
  PNM_AP_CMD_GET_ARVENDORBLOCK_RESPONSE_REQ          = 0x00009420,
  PNM_AP_CMD_GET_ARVENDORBLOCK_RESPONSE_CNF          = 0x00009421,
  PNM_AP_CMD_READ_RECORD_SUBM_REQ                    = 0x00009422,
  PNM_AP_CMD_READ_RECORD_SUBM_CNF                    = 0x00009423,
  PNM_AP_CMD_WRITE_RECORD_SUBM_REQ                   = 0x00009424,
  PNM_AP_CMD_WRITE_RECORD_SUBM_CNF                   = 0x00009425,
  PNM_AP_CMD_READ_IMPLICIT_RECORD_REQ                = 0x00009426,
  PNM_AP_CMD_READ_IMPLICIT_RECORD_CNF                = 0x00009427,
  PNM_AP_CMD_RECV_ALARM_IND                          = 0x00009428,
  PNM_AP_CMD_RECV_ALARM_RES                          = 0x00009429,
  PNM_AP_CMD_RECV_DIAGNOSIS_IND                      = 0x0000942A,
  PNM_AP_CMD_RECV_DIAGNOSIS_RES                      = 0x0000942B,
  PNM_AP_CMD_ACK_ALARM_REQ                           = 0x0000942C,
  PNM_AP_CMD_ACK_ALARM_CNF                           = 0x0000942D,
  PNM_AP_CMD_SET_ARSTATUS_REQ                        = 0x0000942E,
  PNM_AP_CMD_SET_ARSTATUS_CNF                        = 0x0000942F,

  /* dcp services */
  PNM_AP_CMD_DCP_SET_NAME_REQ                        = 0x00009430,
  PNM_AP_CMD_DCP_SET_NAME_CNF                        = 0x00009431,
  PNM_AP_CMD_DCP_SET_IP_REQ                          = 0x00009432,
  PNM_AP_CMD_DCP_SET_IP_CNF                          = 0x00009433,
  PNM_AP_CMD_DCP_SET_SIGNAL_REQ                      = 0x00009434,
  PNM_AP_CMD_DCP_SET_SIGNAL_CNF                      = 0x00009435,
  PNM_AP_CMD_DCP_RESET_FACTORY_SETTINGS_REQ          = 0x00009436,
  PNM_AP_CMD_DCP_RESET_FACTORY_SETTINGS_CNF          = 0x00009437,
  PNM_AP_CMD_DCP_GET_REQ                             = 0x00009438,
  PNM_AP_CMD_DCP_GET_CNF                             = 0x00009439,

  PNM_AP_CMD_DCP_SET_IND                             = 0x0000943E,
  PNM_AP_CMD_DCP_SET_RES                             = 0x0000943F,

  /* miscellanous controller services */
  PNM_AP_CMD_STORE_REMANENT_IND                      = 0x00009440,
  PNM_AP_CMD_STORE_REMANENT_RES                      = 0x00009441,
  PNM_AP_CMD_LOAD_REMANENT_REQ                       = 0x00009442,
  PNM_AP_CMD_LOAD_REMANENT_CNF                       = 0x00009443,
  PNM_AP_CMD_ARSTATUS_CHANGED_IND                    = 0x00009444,
  PNM_AP_CMD_ARSTATUS_CHANGED_RES                    = 0x00009445,

  /* get configuration services */
  PNM_AP_CMD_CFG_GET_NUM_CONFIGURED_OBJECTS_REQ      = 0x00009450,
  PNM_AP_CMD_CFG_GET_NUM_CONFIGURED_OBJECTS_CNF      = 0x00009451,

  PNM_AP_CMD_CFG_GET_NUM_OF_RECORDS_REQ              = 0x00009452,
  PNM_AP_CMD_CFG_GET_NUM_OF_RECORDS_CNF              = 0x00009453,

  PNM_AP_CMD_CFG_GET_IOC_REQ                         = 0x00009454,
  PNM_AP_CMD_CFG_GET_IOC_CNF                         = 0x00009455,

  PNM_AP_CMD_CFG_GET_IOC_PRM_LIST_REQ                = 0x00009456,
  PNM_AP_CMD_CFG_GET_IOC_PRM_LIST_CNF                = 0x00009457,
  PNM_AP_CMD_CFG_GET_IOC_PRM_REQ                     = 0x00009458,
  PNM_AP_CMD_CFG_GET_IOC_PRM_CNF                     = 0x00009459,

  PNM_AP_CMD_CFG_GET_IOD_REQ                         = 0x0000945A,
  PNM_AP_CMD_CFG_GET_IOD_CNF                         = 0x0000945B,

  PNM_AP_CMD_CFG_GET_AR_PRM_LIST_REQ                 = 0x0000945C,
  PNM_AP_CMD_CFG_GET_AR_PRM_LIST_CNF                 = 0x0000945D,
  PNM_AP_CMD_CFG_GET_AR_PRM_REQ                      = 0x0000945E,
  PNM_AP_CMD_CFG_GET_AR_PRM_CNF                      = 0x0000945F,

  PNM_AP_CMD_CFG_GET_IOCR_REQ                        = 0x00009460,
  PNM_AP_CMD_CFG_GET_IOCR_CNF                        = 0x00009461,

  PNM_AP_CMD_CFG_GET_SUBMODULE_REQ                   = 0x00009462,
  PNM_AP_CMD_CFG_GET_SUBMODULE_CNF                   = 0x00009463,
  PNM_AP_CMD_CFG_GET_RECORD_REQ                      = 0x00009464,
  PNM_AP_CMD_CFG_GET_RECORD_CNF                      = 0x00009465,

  /* add other GET CONFIG commands here */

  /* diagnostic services */
  PNM_AP_CMD_GET_LOGBOOK_REQ                         = 0x000094F0,
  PNM_AP_CMD_GET_LOGBOOK_CNF                         = 0x000094F1,

};

/** pnm command code definition */
typedef enum PNM_AP_CMD_Etag PNM_AP_CMD_E;

/******************************************************************************

  ,ad8888ba,                               ad88  88
 d8"'    `"8b                             d8"    ""
d8'                                       88
88              ,adPPYba,   8b,dPPYba,  MM88MMM  88   ,adPPYb,d8
88             a8"     "8a  88P'   `"8a   88     88  a8"    `Y88
Y8,            8b       d8  88       88   88     88  8b       88
 Y8a.    .a8P  "8a,   ,a8"  88       88   88     88  "8a,   ,d88
  `"Y8888Y"'    `"YbbdP"'   88       88   88     88   `"YbbdP"Y8
                                                      aa,    ,88
                                                       "Y8bbdP"

patorjk.com/software/taag --> Font Univers
******************************************************************************/

/******************************************************************************
 CONFIG IO-CONTROLLER SERVICE
******************************************************************************/

/** \brief Start mode for the data exchange

  Usable flags in PNM_AP_CFG_IOC_DATA_Ttag::ulSystemFlags
*/
enum PNM_AP_CFG_STARTMODE_FLAGS_Etag
{
  /** CONTROLER will start with the data exchange on the bus after the initialization has been ended. */
  PNM_AP_IOC_FLAG_STARTMODE_AUTOMATIC              = 0x00000000,
  /** Application program must activate the data exchange on the bus. */
  PNM_AP_IOC_FLAG_STARTMODE_APPLICATION_CONTROLLED = 0x00000001,
  /** Controller will force devices to set the configured name of station even if a non empty nameOfStation is set.
   * Please note that this feature is only available if TOPOLOGY INFORMATION is set using PNM_AP_CFG_TOPO_DATA_T */
  PNM_AP_IOC_FLAG_FORCE_NAME_ASSIGNMENT            = 0x00000002,
  /** controller will accept name of station / ip address changed from bus
   * if not in state operate
   */
  PNM_AP_IOC_FLAG_ENABLE_DCP_SET                   = 0x00000004,
};

typedef enum PNM_AP_CFG_STARTMODE_FLAGS_Etag PNM_AP_CFG_STARTMODE_FLAGS_E;

enum PNM_AP_CFG_ALARM_HANDLING_FLAGS_Etag
{
  /* includes diagnosis disappears */
  PNM_AP_CFG_ALARM_HANDLING_FLAG_DIAGNOSIS            = 0x00000001,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_PROCESS              = 0x00000002,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_PULL                 = 0x00000004,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_PLUG                 = 0x00000008,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_PULL_MODULE          = 0x00000010,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_PLUG_WRONG           = 0x00000020,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_STATUS               = 0x00000040,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_UPDATE               = 0x00000080,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_MEDIA_REDUND         = 0x00000100,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_CONTROLLED           = 0x00000200,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_RELEASED             = 0x00000400,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_RETURN_OF_SUBM       = 0x00000800,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_MCR_MISMATCH         = 0x00001000,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_PORT_DATA_CHANGE     = 0x00002000,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_SYNC_DATA_CHANGE     = 0x00004000,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_TIME_DATA_CHANGE     = 0x00008000,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_ISOCHR_MODE_PROBLEM  = 0x00010000,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_NETW_COMP_PROBLEM    = 0x00020000,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_DFP_PROBLEM          = 0x00040000,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_MRPD_PROBLEM         = 0x00080000,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_MULT_INTF_MISMATCH   = 0x00100000,
  PNM_AP_CFG_ALARM_HANDLING_FLAG_UPLOAD_AND_RETRIVAL  = 0x00200000,
  /* unknown alarm types MUST be handled in application and are NOT handled automatically by protocol stack */
  PNM_AP_CFG_ALARM_HANDLING_FLAG_MANUFACTURER_ALARMS  = 0x80000000,
};

/** valid values for DPM watchdog time */
#define PNM_AP_DPM_WATCHDOG_TIME_OFF     0x00000000L /* Watchdog supervision disabled          */
#define PNM_AP_DPM_WATCHDOG_TIME_MIN     0x00000014L /* Minimum value for watchdog supervision */
#define PNM_AP_DPM_WATCHDOG_TIME_MAX     0x0000ffffL /* Maximum value for watchdog supervision */


typedef enum PNM_AP_CFG_ALARM_HANDLING_FLAGS_Etag PNM_AP_CFG_ALARM_HANDLING_FLAGS_E;

/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_IOC_STRUCT_VERSION_1    (0x0001)
#define PNM_AP_CFG_IOC_STRUCT_VERSION_2    (0x0002)

/* request packet data */
typedef struct PNM_AP_CFG_IOC_DATA_Ttag PNM_AP_CFG_IOC_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_IOC_DATA_Ttag
{
  /** structure version of this structure */
  uint32_t              ulStructVersion;
  /** The definition of the ulSystemFlags are \ref PNM_AP_CFG_STARTMODE_FLAGS_Etag */
  uint32_t              ulSystemFlags;
  /** Automatic Alarm handling flags

      With the ulAlarmHandlingFlags you can define which alarms should be handled by the fieldbus stack automatically.
      The definition of the ulAlarmHandlingFlags are \ref PNM_AP_CFG_ALARM_HANDLING_FLAGS_Etag enumeration.
  */
  uint32_t              ulAlarmHandlingFlags;
  /** DPM Channel Watchdog time in ms

    \arg \c Deactivated 0
    \arg \c Min 0
    \arg \c Max 65535)
  */
  uint32_t              ulDpmWatchdogTime;
  /** VendorID to be used by IO Controller
        \note This parameter is for the OEMs to handover the own VendorID number.
        \par See Spec:
        Coding of fields related to Instance, DeviceID, VendorID
  */
  uint16_t              usVendorID;
  /** DeviceID to be used by IO Controller
        \note This parameter is for the OEMs to handover the Vendor spezific DeviceID number.
        \par See Spec:
        Coding of fields related to Instance, DeviceID, VendorID
  */
  uint16_t              usDeviceID;
  /** IP address to be used by IO Controller */
  uint32_t              ulIPAddr;
  /** subnet mask to be used by IO Controller */
  uint32_t              ulNetmask;
  /** Gateway address to be used by IO Controller */
  uint32_t              ulGateway;
  /** Device Type to be used by IO Controller, Pad With Zero:
      we don't need a length here, can be computed by searching for zeros
      \par See Spec:
       See Coding of the field DeviceType
  */
  uint8_t               abDeviceType[25];
  /** NameOfStation to be used by IO Controller, Pad With Zero:
      we don't need a length here, can be computed by searching for zeros
      \note The default name should be get from the GSDML attribute \c DNS_CompatibleName.
      \par See Spec:
       Coding of the field NameOfStationValue
  */
  uint8_t               abNameOfStation[240];
  /** OrderId to be used by IO Controller, Pad With Zero:
      we don't need a length here, can be computed by searching for zeros
      \note  The GSDML Element \c OrderNumber.
      \par See Spec:
       Coding of the field OrderID
  */
  uint8_t               abOrderId[20];

  /** Start offset in DPM input area for slave status bitlists.
      Each bitlist has 128 bits (one for each IO Device) thus requires 16 byte.
      The bitlists are always in the fix order: configured, active, faulty

      The bitlist must be located outside of the process data area, that is,
      either before any process data or behind.
  */
  uint16_t              usBitlistStartOffset;
  /** Fix alignment of structure fields due to definition abDeviceType */
  uint8_t               bPadding;
  /** Start offset in DPM input area for the IRT cycle counter
   *
   * This field is available since structure version 2. Set to 0xFFFF
   * to disable cycle counter in input data area. */
  uint16_t              usIoTimingInfoOffset;
};

/* request packet */
typedef struct PNM_AP_CFG_IOC_REQ_Ttag PNM_AP_CFG_IOC_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_IOC_REQ_Ttag
{
  /** packet header */
  PNM_AP_PCK_HEADER_T               tHead;
  /** packet data */
  PNM_AP_CFG_IOC_DATA_T             tData;
} ;

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T          PNM_AP_CFG_IOC_CNF_T;

/** packet union */
typedef union PNM_AP_CFG_IOC_PCK_Ttag PNM_AP_CFG_IOC_PCK_T;
union PNM_AP_CFG_IOC_PCK_Ttag
{
  /** request packet */
  PNM_AP_CFG_IOC_REQ_T      tReq;
  /** confirmation packet */
  PNM_AP_CFG_IOC_CNF_T      tCnf;
};

/*******************************************************************************
 CONFIGURE IO-CONTROLLER PARAMETERS
*******************************************************************************/

/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_IOC_PRM_SYNC_STRUCT_VERSION_1    (0x0001)
#define PNM_AP_CFG_IOD_PRM_PDSYNCDATA_VERSION_1     (0x0001)

enum PNM_AP_CFG_PRM_SYNC_SYNCPROP_FLAGS_Etag
{
  /** IO Controller shall be a SyncSlave, currently not supported by the firmware */
  PNM_AP_CFG_PRM_SYNC_SYNCPROP_ROLE_SLAVE  = 0x00000001,
  /** IO Controller shall be the SyncMaster */
  PNM_AP_CFG_PRM_SYNC_SYNCPROP_ROLE_MASTER = 0x00000002,
};

typedef enum PNM_AP_CFG_PRM_SYNC_SYNCPROP_FLAGS_Etag PNM_AP_CFG_PRM_SYNC_SYNCPROP_FLAGS_E;


/* IRT related parameters */
typedef struct PNM_AP_CFG_PRM_SYNC_Ttag PNM_AP_CFG_PRM_SYNC_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_SYNC_Ttag
{
  /** structure version of this structure */
  uint32_t                ulStructVersion;
  /** SyncDomain UUID to be used by IO Controllers SyncMaster implementation,
   *   The same value needs to be used for all IO Devices that have an IRT connection to this IO Controller
   *   \par See Spec:
   *    Coding of the field PTCP_SubdomainUUID
   *
   *  The value of this UUID shall be generated at the time the sync configuration
   *  parameters are created. That means if an existing configuration is modified but none
   *  of the parameters is changed it is recommended to not change the uuid.
   */
  PNM_UUID_T              tSyncDomainUuid;
  /** SyncProperties, see \ref PNM_AP_CFG_PRM_SYNC_SYNCPROP_FLAGS_E
      \par See Spec:
       Coding of the field SyncProperties
  */
  uint16_t                usSyncProperties;
  /** irt sendclock (bus cycle) to be used by IO Controllers SyncMaster implementation
   *  needs to match the SendClockFactor of all IRT IO Devices
   *  \par See Spec:
   *   Coding of the field SendClockFactor
   *
   *   This value defines the bus cycle. It is specified in multiples of 31.25 us.
   *   The bus cycle can be configured by the user and is limited to values between 31.25 us and
   *   4ms. (For Hilscher from 250us to 4ms) Typically we have selection of 250us, 500us, 1ms,
   *   2ms and 4ms. The possible bus cycle is limited by the capabilities of attached devices
   *   (GSDML) and the number of devices and amount I/O Data. E.g if running on 250us we only have
   *   125us for IRT process data. A 64 Byte frame takes about 6us and some security margins.
   */
  uint16_t                usSendClockFactor;
  /** PllWindow Sync Slave specific parameter
   *  \par See Spec:
   *   Coding of the field PLLWindow
   *
   *   This value is not relevant if running in sync master mode. If running in
   *   sync slave mode this value describes the maximum allowed jitter between
   *   the devices own clock and the clock from sync frame. For slave devices this value
   *   can be retrieved from GSDML.
   */
  uint32_t                ulPllWindow;
  /** syncframe send cycle time to be used by IO Controllers SyncMaster implementation
   *  \par See Spec:
   *    Coding of the field SyncSendFactor
   *
   *  This value defines the interval of sending sync frames. It uses similar coding as
   *  the send clock (multiples of 31.25 us). Historically the value is set to
   *  30 times the value of the send clock factor: E.g. Buscycle of 1ms results in a
   *  Sync send factor 1ms / 31.25 us * 30 = 960
   */
  uint32_t                ulSyncSendFactor;
  /** Sync Slave specific timeout parameter
   *  \par See Spec:
   *   Coding of the field PTCPTimeoutFactor
   *
   *   This value is only relevant in sync slave mode. It describes after how much sync
   *   send clock intervals without a valid sync frame the device detects a timeout.
   *
   *   \arg \c Deactivated 0
   *   \arg \c Default 0x0006
   *   \arg \c Min 0
   *   \arg \c Max 0x01FF
   *
   */
  uint16_t                usPtcpTimeoutFactor;
  /** Sync Slave specific timeout parameter
   *  \par See Spec:
   *   Coding of the field PTCPTakeoverTimeoutFactor
   *
   *   \arg \c Deactivated 0
   *   \arg \c Default 0x0003
   *   \arg \c Min 0
   *   \arg \c Max 0x01FF
   */
  uint16_t                usPtcpTakeoverTimeoutFactor;
  /** Sync Slave specific parameter
   *  \par See Spec:
   *   Coding of the field PTCPMasterStartupTime
   *
   *  \arg \c Deactivated 0
   *  \arg \c Default 0x003C
   *  \arg \c Min 0
   *  \arg \c Max 0x012C
   */
  uint16_t                usPtcpMasterStartupTime;
  /** Sync specific Master priority parameter
   *  \par See Spec:
   *   Coding of the field PTCP_MasterPriority1
   *
   *   This value should be assigned a default value by engineering and might
   *   be adjusted by the used according its needs.
   */
  uint8_t                 bPtcpMasterPrio1;
  /** Sync specific Master priority parameter
   *  \par See Spec:
   *   Coding of the field PTCP_MasterPriority2
   *
   *   This value should be assigned a default value by engineering and might
   *   be adjusted by the used according its needs.
   */
  uint8_t                 bPtcpMasterPrio2;
  /** Sync Subdomain name to be used by IO Controllers SyncMaster implementation.
   * Zero padded: No length required, can be calculated by searching the first zero.
   * The same value needs to be used for all IO Devices that have an IRT connection to this IO Controller
   * \par See Spec:
   *  Coding of the field
   *
   *  The value of this parameter is typically assigned a default value by the engineering system.
   *  Nevertheless, the user might choose a different value to allow multiple sync master configured by
   *  different engineering systems in one network.
   */
  uint8_t                 abSubdomainName[240];
} ;

/* Frame Details Bitmask
 *
 * @details This bitmask specifies properties of a IR data frame entry */

enum PNM_AP_CFG_FRAME_DETAILS_Etag
{
  PNM_AP_CFG_FRAME_NO_SYSNC_FRAME        = 0x00,
  /** This frame is a primary sync frame */
  PNM_AP_CFG_FRAME_PRIMARY_SYNC_FRAME    = 0x01,
  /** This frame is a secondary sync frame */
  PNM_AP_CFG_FRAME_SECONDARY_SYNC_FRAME  = 0x02,
  /** this frame shall be monitored by mrpd */
  PNM_AP_CFG_FRAME_MONITOR_MRPD          = 0x80,
};

typedef enum PNM_AP_CFG_FRAME_DETAILS_Etag PNM_AP_CFG_FRAME_DETAILS_E;

/* allowed values for field ulStructVersion for controller */
#define PNM_AP_CFG_IOC_PRM_IRDATA_FRAME_STRUCT_VERSION_1    (0x0001)

/* allowed values for field ulStructVersion for device */
#define PNM_AP_CFG_IOD_PRM_IRDATA_FRAME_STRUCT_VERSION_1    (0x0001)

typedef struct PNM_AP_CFG_PRM_IRDATA_FRAME_Ttag PNM_AP_CFG_PRM_IRDATA_FRAME_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_IRDATA_FRAME_Ttag
{
  /** structure version of this structure */
  uint32_t                ulStructVersion;

  uint16_t                usEntryId;
  /** FrameID of this frame
     needs to match the FrameID of the IOCR of the associated IO Device
     \par See Spec:
      Coding of the field FrameID
  */
  uint16_t                usFrameId;
  /** length of this frame
     needs to match the length of the frame of the IOCR of the associated IO Device */
  uint16_t                usCSDULength;
  /** Phase when this frame shall be transmitted
     needs to match the Phase of the IOCR of the associated IO Device
     \par See Spec:
      Coding of the field Phase
  */
  uint16_t                usPhase;
  /** ReductionRatio of this frame
      needs to match the Phase of the IOCR of the associated IO Device
      \par See Spec:
       Coding of the field ReductionRatio
  */
  uint16_t                usReduction;
  /** Properties of the Frame, see PNM_AP_CFG_FRAME_DETAILS_E
      \par See Spec:
     Coding of the field FrameDetails
   */
  uint8_t                 bFrameDetails;
  /** Received Port of the Frame
     \par See Spec:
      Coding of the field RxPort
  */
  uint8_t                 bRxPort;
  /** FrameSendOffset, point in time when transmitting the frame shall be done in ns
      \par See Spec:
       Coding of the field FrameSendOffset
  */
  uint32_t                ulFrameSendOffset;
  /**  */
  uint8_t                 atTxPortGroups[16];
};

#define PNM_AP_CFG_PRM_MAX_BEGIN_END_ASSIGNMENTS         (16)


typedef struct PNM_AP_CFG_PRM_BEGIN_END_ASSIGNMENT_Ttag PNM_AP_CFG_PRM_BEGIN_END_ASSIGNMENT_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_BEGIN_END_ASSIGNMENT_Ttag
{
  /** Red Phase begin time for Transmit in nanoseconds
      \par See Spec:
       Coding of the field RedOrangePeriodBegin
  */
  uint32_t     ulTxRedBegin;
  /** Green Phase begin time for Transmit in nanoseconds
      \par See Spec:
       Coding of the field GreenPeriodBegin
  */
  uint32_t     ulTxGreenBegin;
  /** Red Phase begin time for Receive in nanoseconds
      \par See Spec:
       Coding of the field RedOrangePeriodBegin
  */
  uint32_t     ulRxRedBegin;
  /** Green Phase begin time for Receive in nanoseconds
      \par See Spec:
       Coding of the field GreenPeriodBegin
  */
  uint32_t     ulRxGreenBegin;
};

/* allowed values for field ulStructVersion for controller */
#define PNM_AP_CFG_IOC_PRM_IRDATA_PHASES_STRUCT_VERSION_1    (0x0001)

/* allowed values for field ulStructVersion for device */
#define PNM_AP_CFG_IOD_PRM_IRDATA_PHASES_STRUCT_VERSION_1    (0x0001)


typedef struct PNM_AP_CFG_PRM_IRDATA_PHASES_Ttag PNM_AP_CFG_PRM_IRDATA_PHASES_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_IRDATA_PHASES_Ttag
{
  /** structure version of this structure */
  uint32_t                                  ulStructVersion;
  /** Phase Assignment, the number of elements inside the array is implicitly defined by the packet headers packet length */
  /* A maximum of \ref PNM_AP_CFG_PRM_MAX_BEGIN_END_ASSIGNMENTS elements is allowed */
  PNM_AP_CFG_PRM_BEGIN_END_ASSIGNMENT_T     atBeginEnd[ARRAYS_OF_LENGTH_ZERO];
} ;


#define PNM_AP_CFG_PRM_MAX_PORTS                             (4)


/* allowed values for field ulStructVersion for controller */
#define PNM_AP_CFG_IOC_PRM_IRDATA_GLOBAL_STRUCT_VERSION_1    (0x0001)

/* allowed values for field ulStructVersion for device */
#define PNM_AP_CFG_IOD_PRM_IRDATA_GLOBAL_STRUCT_VERSION_1    (0x0001)
#define PNM_AP_CFG_IOD_PRM_IRDATA_GLOBAL_STRUCT_VERSION_2    (0x0002)


/** Definition of Global IRData Frame Properties
 *
 * @details Only one define of each group shall be used at the
 * same time. The allowed values depend on the features of the device
 * and can be derived from GSDML file */
enum PNM_AP_CFG_FRAME_DATA_PROP_Etag
{
  /* The device shall use absolute forwarding mode */
  PNM_AP_CFG_FRAME_PROP_FORWARDING_MODE_ABSOLUTE   = 0x00000000,
  /* The device shall use relative forwarding mode */
  PNM_AP_CFG_FRAME_PROP_FORWARDING_MODE_RELATIVE   = 0x00000001,

  PNM_AP_CFG_FRAME_PROP_FAST_FORWARDING_LEGACY     = 0x00000000,
  PNM_AP_CFG_FRAME_PROP_FAST_FORWARDING_RTC3_MCAST = 0x00000002,
  PNM_AP_CFG_FRAME_PROP_FAST_FORWARDING_MCAST      = 0x00000004,

  PNM_AP_CFG_FRAME_PROP_FRAGMENTATION_OFF          = 0x00000000,
  PNM_AP_CFG_FRAME_PROP_FRAGMENTSIZE_128           = 0x00000008,
  PNM_AP_CFG_FRAME_PROP_FRAGMENTSIZE_265           = 0x00000010,
};

typedef enum PNM_AP_CFG_FRAME_DATA_PROP_Etag PNM_AP_CFG_FRAME_DATA_PROP_E;


typedef struct PNM_AP_CFG_PRM_IRDATA_PORT_Ttag PNM_AP_CFG_PRM_IRDATA_PORT_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_IRDATA_PORT_Ttag
{
  /** Maximum Tx Delay of Port */
  uint32_t ulMaxPortTxDelay;
  /** Maximum Rx Delay of Port */
  uint32_t ulMaxPortRxDelay;
  /** Maximum Rx Line Delay */
  uint32_t ulMaxLineRxDelay;
  /** Maximum */
  uint32_t ulYellowTime;
};

typedef struct PNM_AP_CFG_PRM_IRDATA_GLOBAL_Ttag PNM_AP_CFG_PRM_IRDATA_GLOBAL_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_IRDATA_GLOBAL_Ttag
{
  /** structure version of this structure */
  uint32_t                     ulStructVersion;
  /** IR Data UUID */
  PNM_UUID_T                   tIRDataUuid;
  /** Maximum Bridge Delay */
  uint32_t                     ulMaxBridgeDelay;
  /** StartOfRed */
  uint16_t                     usStartOfRedFrameId;
  /** EndOfRed */
  uint16_t                     usEndOfRedFrameId;
  /** Number of IRT Frame data sets for this frame */
  uint16_t                     usNumFrames;
  /** number of Ports */
  uint8_t                      bNumPorts;
  /** number of Phases (begin end assignments) */
  uint8_t                      bNumPhases;
  /** Set according to \ref PNM_AP_CFG_FRAME_DATA_PROP_E,
   *  At most one define of each group shall be set */
  uint32_t                     ulFrameDataProperties;
  /** Port Configuration, shall contain bNumPorts entries */
  PNM_AP_CFG_PRM_IRDATA_PORT_T atPorts[ARRAYS_OF_LENGTH_ZERO];
} ;

typedef enum
{
  PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_ENABLE_FLAG_DOMAIN_BOUNDARY        = 0x0001,
  PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_ENABLE_FLAG_MULTICAST_BOUNDARY     = 0x0002,
  PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_ENABLE_FLAG_PEERTOPEER_BOUNDARY    = 0x0004,
  PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_ENABLE_FLAG_DCP_BOUNDARY           = 0x0008,
  PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_ENABLE_FLAG_MAUTYPE                = 0x0010,
  PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_ENABLE_FLAG_LINKSTATE              = 0x0020,
  PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_ENABLE_FLAG_PREAMBLELENGTH         = 0x0040,
} PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_ENABLE_FLAGS;


typedef enum
{
  PNM_AP_CFG_PRM_PEER_TO_PEER_BOUNDARY_PASS                 = 0x0000,
  PNM_AP_CFG_PRM_PEER_TO_PEER_BOUNDARY_BLOCK_LLDP           = 0x0001,
  PNM_AP_CFG_PRM_PEER_TO_PEER_BOUNDARY_BLOCK_PTCP_DELAY     = 0x0002,
  PNM_AP_CFG_PRM_PEER_TO_PEER_BOUNDARY_BLOCK_PTCP_PATH      = 0x0004,
} PNM_AP_CFG_PRM_PEER_TO_PEER_BOUNDARY_E;

typedef enum
{
  PNM_AP_CFG_PRM_DCP_BOUNDARY_PASS               = 0x0000,
  PNM_AP_CFG_PRM_DCP_BOUNDARY_BLOCK_IDENTIFY     = 0x0001,
  PNM_AP_CFG_PRM_DCP_BOUNDARY_BLOCK_HELLO        = 0x0002,
} PNM_AP_CFG_PRM_DCP_BOUNDARY_E;

enum PNM_AP_CFG_PRM_PORT_MAUTYPE_Etag
{
  /* Disable Port */
  PNM_AP_CFG_IOC_PRM_PORT_MAUTYPE_PORT_DISABLED  = 0,
  /** use fix port setting for copper, 100 MBit/s FullDuplex */
  PNM_AP_CFG_IOC_PRM_PORT_MAUTYPE_100BASETX      = 16,
};
typedef enum PNM_AP_CFG_PRM_PORT_MAUTYPE_Etag PNM_AP_CFG_PRM_PORT_MAUTYPE_E;

enum PNM_AP_CFG_PRM_PORT_PREAMBLE_LENGTH_Etag
{
  /** preamble length of 7 octets (default) */
  PNM_AP_CFG_IOC_PRM_PORT_PREAMBLE_LENGTH_7_OCTETS = 7,
  /** use short preamble of 1 octet */
  PNM_AP_CFG_IOC_PRM_PORT_PREAMBLE_LENGTH_1_OCTET  = 1,
};
typedef enum PNM_AP_CFG_PRM_PORT_PREAMBLE_LENGTH_Etag PNM_AP_CFG_PRM_PORT_PREAMBLE_LENGTH_E;

#define PNM_AP_CFG_PRM_PDPORTDATAADJUST_VERSION_1      (0x0001)

typedef struct PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_Ttag PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructversion;
  /** field containing the valid parameters of this packet - see above */
  uint32_t ulEnableFlag;
  /** indication of Domain boundary Ingress
      \par See Spec:
       Coding of the field DomainBoundaryIngress
   * */
  uint32_t ulDomainBoundaryIngress;
  /** indication of Domain boundary Ergress
      \par See Spec:
       Coding of the field DomainBoundaryEgress
   * */
  uint32_t ulDomainBoundaryEgress;
  /** indication of Multicast boundary
      \par See Spec:
       Coding of the field MulticastBoundary
   * */
  uint32_t ulMulticastBoundary;
  /** indication of peer to peer boundary see PNM_AP_CFG_PRM_PEER_TO_PEER_BOUNDARY_E
      \par See Spec:
       Coding of the field PeerToPeerBoundary
   * */
  uint32_t ulPeerToPeerBoundary;

  /** indication of DCP boundary see PNM_AP_CFG_PRM_DCP_BOUNDARY_E
      \par See Spec:
       Coding of the field DCPBoundary
   * */
  uint32_t ulDCPBoundary;
  /** mau type to use by IO Device
      \par See Spec:
       Coding of the field MAUType
   * */
  uint16_t usMAUType;
  /** link state to use by IO Device
      \par See Spec:
       Coding of the field LinkState
   * */
  uint16_t usLinkState;
  /** Preamble length to use by IO Device
      \par See Spec:
       Coding of the field PreambleLength
   * */
  uint16_t usPreambleLength;
};



/* port setting related parameters */
typedef enum
{
  PNM_AP_CFG_PRM_PDPORTDATA_CHECK_ENABLE_FLAG_PEER_ID             = 0x0001,
  PNM_AP_CFG_PRM_PDPORTDATA_CHECK_ENABLE_FLAG_LINEDELAY           = 0x0002,
  PNM_AP_CFG_PRM_PDPORTDATA_CHECK_ENABLE_FLAG_MAUTYPE             = 0x0004,
  PNM_AP_CFG_PRM_PDPORTDATA_CHECK_ENABLE_FLAG_LINKSTATE           = 0x0008,
  PNM_AP_CFG_PRM_PDPORTDATA_CHECK_ENABLE_FLAG_SYNCMODE            = 0x0010,
  PNM_AP_CFG_PRM_PDPORTDATA_CHECK_ENABLE_FLAG_MAUTYPEMODE         = 0x0020,
} PNM_AP_CFG_PRM_PDPORTDATA_CHECK_ENABLE_FLAGS;

#define PNM_AP_CFG_PRM_PDPORTDATACHECK_VERSION_1      (0x0001)

typedef struct PNM_AP_CFG_PRM_PDPORTDATA_CHECK_Ttag PNM_AP_CFG_PRM_PDPORTDATA_CHECK_T;

 __PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDPORTDATA_CHECK_Ttag
{
   /** structure version of this structure */
  uint32_t ulStructversion;
  /** field containing the valid parameters of this packet - see above */
  uint32_t ulEnableFlag;
  /** Remote port name to be used by Device CHK_REM state machine
      Pad With Zero: we don't need a length here, can be computed by searching for zeros
      \par See Spec:
       Coding of the field PeerPortName
  */
  /** Id of remote peer expected at this port. Combinatin of Port Id and Nameofstation:
   *  "<port-xyz[-abcdef]>.<nameofstation>"
      Pad With Zero: we don't need a length here, can be computed by searching for zeros
      \par See Spec:
       Coding of the field PeerPortName
  */
  uint8_t  abPeerId[255];

  uint8_t  bReserved;
  /** the value of the line delay between local and peer port to be used by IO Device CHK_REM state machine
      \par See Spec:
       Coding of the field LineDelay
  */
  uint32_t ulLineDelay;
  /** mau type to use by Device CHK_REM state machine
      \par See Spec:
       Coding of the field MAUType
   */
  uint16_t usMauType;
  /** link state to use by  Device CHK_REM state machine
      \par See Spec:
       Coding of the field LinkState
   **/
  uint16_t usSyncMode;
};

/* the Max possible value of MRP Topology Change Interval */
#define PNM_AP_CFG_MRP_TOPO_CHANGE_INTERVAL_MAX         (100)

/* the possible values of MRP Topology Change Repeat Count */
#define PNM_AP_CFG_MRP_TOPO_CHANGE_REPEAT_COUNT_MIN     (1)
#define PNM_AP_CFG_MRP_TOPO_CHANGE_REPEAT_COUNT_MAX     (5)

/* the possible values of MRP Short Test Interval */
#define PNM_AP_CFG_MRP_SHORT_TEST_INTERVAL_MIN          (1)
#define PNM_AP_CFG_MRP_SHORT_TEST_INTERVAL_MAX          (500)

/* the possible values of MRP Default Test Interval */
#define PNM_AP_CFG_MRP_DEFAULT_TEST_INTERVAL_MIN        (1)
#define PNM_AP_CFG_MRP_DEFAULT_TEST_INTERVAL_MAX        (1000)

/* the possible values of MRP Monitoring Count */
#define PNM_AP_CFG_MRP_TEST_MONITOR_COUNT_MIN           (2)
#define PNM_AP_CFG_MRP_TEST_MONIROT_COUNT_MAX           (10)

typedef struct PNM_AP_CFG_PRM_MRP_MANAGER_PARAMS_Ttag PNM_AP_CFG_MRP_MANAGER_PARAMS_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_MRP_MANAGER_PARAMS_Ttag
{
  /** MRP Manager priority
     \par See Spec:
      Coding of the field MRP_Prio
  */
  uint16_t usMRP_Prio;
  /** MRP Topology Change Interval in milliseconds,
   *  at which the ring's master node sends MRP Chnage frames.
     \par See Spec:
      Coding of the field Phase
  */
  uint16_t usMRP_TOPchgT;
  /** MRP Topology Change Repeat Count specifies the number of times the MRP Topology Change frame should repeat
     \par See Spec:
      Coding of the field MRP_TOPNRmax
  */
  uint16_t usMRP_TOPNRmax;
  /** MRP Short Test Interval in milliseconds,
   *  at which the ring's master node sends MRP Test frames after link changes in the ring.
     \par See Spec:
      Coding of the field MRP_TSTshortT
  */
  uint16_t usMRP_TSTshortT;
  /** MRP default Test Interval in milliseconds,
   *  at which the ring's master node sends MRP Test frames after link changes in the ring.
     \par See Spec:
      Coding of the field MRP_TSTdefaultT
  */
  uint16_t usMRP_TSTdefaultT;
  /** MRP Test Monitoring Count specifies the number
     \par See Spec:
      Coding of the field MRP_TSTNRmax
  */
  uint16_t usMRP_TSTNRmax;
};

/* the possible values of Link Down Interval */
#define PNM_AP_CFG_MRP_LINK_DOWN_INTERVAL_MIN          (1)
#define PNM_AP_CFG_MRP_LINK_DOWN_INTERVAL_MAX          (1000)

/* the possible values of Link Up Interval */
#define PNM_AP_CFG_MRP_LINK_UP_INTERVAL_MIN            (1)
#define PNM_AP_CFG_MRP_LINK_UP_INTERVAL_MAX            (1000)

/* the possible values of MRP Link Change Count */
#define PNM_AP_CFG_MRP_LINK_CHANGE_COUNT_MIN           (1)
#define PNM_AP_CFG_MRP_LINK_CHANGE_COUNT_MAX           (5)

typedef struct PNM_AP_CFG_PRM_MRP_CLIENT_PARAMS_Ttag PNM_AP_CFG_MRP_CLIENT_PARAMS_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_MRP_CLIENT_PARAMS_Ttag
{
  /** MRP Link Down Interval
     \par See Spec:
     Coding of the field MRP_LNKdownT
  */
  uint16_t usMRP_LNKdownT;
  /** MRP Link Up Interval
     \par See Spec:
      Coding of the field MRP_LNKupT
  */
  uint16_t usMRP_LNKupT;
  /** MRP Link Change Count
     \par See Spec:
      Coding of the field MRP_LNKNRmax
  */
  uint16_t usMRP_LNKNRmax;
};

typedef struct PNM_AP_CFG_PRM_MRP_AUTOMANAGER_PARAMS_Ttag PNM_AP_CFG_MRP_AUTOMANAGER_PARAMS_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_MRP_AUTOMANAGER_PARAMS_Ttag
{
  PNM_AP_CFG_MRP_MANAGER_PARAMS_T tManager;
  PNM_AP_CFG_MRP_CLIENT_PARAMS_T  tClient;
};

typedef union
{
  PNM_AP_CFG_MRP_MANAGER_PARAMS_T     tManager;
  PNM_AP_CFG_MRP_CLIENT_PARAMS_T      tClient;
} PNM_AP_CFG_MRP_PARAMS_V1_T;

/** For automanager we need both information */
typedef struct
{
  PNM_AP_CFG_MRP_MANAGER_PARAMS_T     tManager;
  PNM_AP_CFG_MRP_CLIENT_PARAMS_T      tClient;
} PNM_AP_CFG_MRP_PARAMS_V2_T;

/* values for MRP Role */
enum PNM_AP_CFG_MRP_ROLE_Etag
{
  /** disable MRP */
  PNM_AP_CFG_MRP_ROLE_DISABLE      = 0,
  /** enable MRP Client */
  PNM_AP_CFG_MRP_ROLE_CLIENT       = 1,
  /** enable MRP Manager */
  PNM_AP_CFG_MRP_ROLE_MANAGER      = 2,
  /** enable MRP Automanager */
  PNM_AP_CFG_MRP_ROLE_AUTOMANAGER  = 3,
};

#define PNM_AP_CFG_PRAM_MRP_VERSION_1      (0x0001)
#define PNM_AP_CFG_PRAM_MRP_VERSION_2      (0x0002)

typedef struct PNM_AP_CFG_PRM_MRP_V1_Ttag PNM_AP_CFG_PRM_MRP_V1_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_MRP_V1_Ttag
{
  /** structure version of this structure
   * use version 1 here */
  uint32_t                ulStructVersion;
  /* MRP instance field will be ignored if ulStructVersion = 1 (no multiple interface ) */
  uint8_t                 bMRP_Instance;
  /** Reserved (Padding) */
  uint8_t                 bReserved;
  /** MRP role see \ref PNM_AP_CFG_MRP_ROLE_Etag
      \par See Spec:
       Coding of the field MRP_Role
   */
  uint16_t                usMrpRole;
  /** MRP Domain UUID to be used by IO Controllers MRP statemachines
      \par See Spec:
       Coding of the field MRP_DomainUUID
  */
  PNM_UUID_T              tMrpDomainUuid;
  /** MRP Domain name to be used by IO Controllers MRP statemachines
      Pad With Zero: we don't need a length here, can be computed by searching for zeros
      \par See Spec:
       Coding of the field MRP_DomainName
  */
  uint8_t                 abMrpDomainName[240];

  /** MRP node parameter data */
  PNM_AP_CFG_MRP_PARAMS_V1_T uParam;
};

typedef struct PNM_AP_CFG_PRM_MRP_V2_Ttag PNM_AP_CFG_PRM_MRP_V2_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_MRP_V2_Ttag
{
  /** structure version of this structure
   * use version 2 here */
  uint32_t                ulStructVersion;
  /* MRP instance field will be ignored if ulStructVersion = 1 (no multiple interface ) */
  uint8_t                 bMRP_Instance;
  /** Reserved (Padding) */
  uint8_t                 bReserved;
  /** MRP role see \ref PNM_AP_CFG_MRP_ROLE_Etag
      \par See Spec:
       Coding of the field MRP_Role
   */
  uint16_t                usMrpRole;
  /** MRP Domain UUID to be used by IO Controllers MRP statemachines
      \par See Spec:
       Coding of the field MRP_DomainUUID
  */
  PNM_UUID_T              tMrpDomainUuid;
  /** MRP Domain name to be used by IO Controllers MRP statemachines
      Pad With Zero: we don't need a length here, can be computed by searching for zeros
      \par See Spec:
       Coding of the field MRP_DomainName
  */
  uint8_t                 abMrpDomainName[240];

  /** MRP node parameter data */
  PNM_AP_CFG_MRP_PARAMS_V2_T tParam;
};
#define PNM_AP_CFG_PRM_MRP_INSTANCE_DATA_VERSION_1             (0x0001)

typedef enum
{
  PNM_AP_CFG_PRM_MRP_CHECK_OFF            = 0,
  PNM_AP_CFG_PRM_MRP_CHECK_MANGER_ON      = 1,
  PNM_AP_CFG_PRM_MRP_CHECK_UUID_ON        = 2,
  PNM_AP_CFG_PRM_MRP_CHECK_MAX            = 4,
} PNM_AP_CFG_PRM_MRP_CHECK_FLAGS_E;

typedef struct PNM_AP_CFG_PRM_MRP_INSTANCE_DATACHECK_Ttag PNM_AP_CFG_PRM_MRP_INSTANCE_DATACHECK_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_MRP_INSTANCE_DATACHECK_Ttag
{
  /** structure version of this structure */
  uint32_t   ulStructVersion;
  /* MRP instance field should be ignored if ulStructVersion = 0  (no multiple interface ) */
  uint8_t    bMRP_Instance;
  /** Reserved (Padding) */
  uint8_t    bReserved1;
  /** Reserved (Padding) */
  uint16_t   usReserved1;
  /** MRP check flags see PNM_AP_CFG_PRM_MRP_CHECK_FLAGS_E
      \par See Spec:
  */
  uint32_t   ulMRP_Check;

  PNM_UUID_T tMRP_DomainUUID;
};

#define PNM_AP_CFG_PRM_PDINFMRPDATACHECK_VERSION_1            (0x0001)
#define PNM_AP_CFG_PRM_PDINFMRPDATACHECK_VERSION_2            (0x0002)

typedef struct PNM_AP_CFG_PRM_PDINTFMRPDATACHECK_Ttag PNM_AP_CFG_PRM_PDINTFMRPDATACHECK_T;

#define PNM_AP_CFG_PRM_MAX_MRP_INSTANCES                        (2)

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDINTFMRPDATACHECK_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;
  /** array with mrp check parameters. Length of array derived from packet length */
  PNM_AP_CFG_PRM_MRP_INSTANCE_DATACHECK_T  atCheck[ARRAYS_OF_LENGTH_ZERO];
};


/** Use atAdjust with PDIntfMRPDataAdjust Blockversion 1.0 */
#define PNM_AP_CFG_PRM_PDINFMRPDATAADJUST_VERSION_1           (0x0001)
/** Use atAdjust with PDIntfMRPDataAdjust Blockversion 1.1 */
#define PNM_AP_CFG_PRM_PDINFMRPDATAADJUST_VERSION_2           (0x0002)
/** Use atAdjustV2 with PDIntfMRPDataAdjust Blockversion 1.1
 *
 * Required for MRP Automanager*/
#define PNM_AP_CFG_PRM_PDINFMRPDATAADJUST_VERSION_3           (0x0003)

typedef struct PNM_AP_CFG_PRM_PDINTFMRPDATAADJUST_Ttag PNM_AP_CFG_PRM_PDINTFMRPDATAADJUST_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDINTFMRPDATAADJUST_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;

  /** array with mrp parameters. Length of array derived from packet length */
  union {
    PNM_AP_CFG_PRM_MRP_V1_T  atAdjust[PNM_AP_CFG_PRM_MAX_MRP_INSTANCES];
    PNM_AP_CFG_PRM_MRP_V2_T  atAdjustV2[PNM_AP_CFG_PRM_MAX_MRP_INSTANCES];
  };
};

/* use version 1 to generate PDPortDataAdjust blockversion 1.0 */
#define PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_VERSION_1           (0x0001)
/* use version 2 to generate PDPortDataAdjust blockversion 1.1 */
#define PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_VERSION_2           (0x0002)

/* data structure for StructVersion 2 */
typedef struct PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_DATA_V2_Ttag PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_DATA_V2_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_DATA_V2_Ttag
{
  PNM_UUID_T tMRP_DomainUUID;
  uint8_t    bMRP_Instance;
};

/* data structure for StructVersion 1 */
typedef struct PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_DATA_V1_Ttag PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_DATA_V1_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_DATA_V1_Ttag
{
  PNM_UUID_T tMRP_DomainUUID;
};

typedef struct PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_Ttag PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_Ttag
{
  /** structure version of this structure */
  uint32_t   ulStructVersion;

  union {
    PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_DATA_V1_T  tPortMrpAdjustV1;
    PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_DATA_V2_T  tPortMrpAdjustV2;
  } uData;
};



#define PNM_AP_CFG_PRM_PDINTFDATAADJUST_VERSION_1   (0x0001)

typedef struct PNM_AP_CFG_PRM_PDINTERFACEADJUST_Ttag PNM_AP_CFG_PRM_PDINTERFACEADJUST_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDINTERFACEADJUST_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;
  /** Multiple Interface mode.
      \par See Spec:
      Coding of the field MultipleInterfaceMode
  */
  uint32_t ulMultipleInterfaceMode;
};

typedef enum
{
  PNM_AP_CFG_PRM_NC_DROP_BUDGET_CHECK_VALID_FLAG_MAINTENANCE_REQUIRED = 0x00000001,
  PNM_AP_CFG_PRM_NC_DROP_BUDGET_CHECK_VALID_FLAG_MAINTENANCE_DEMANDED = 0x00000002,
  PNM_AP_CFG_PRM_NC_DROP_BUDGET_CHECK_VALID_FLAG_ERROR                = 0x00000004,
} PNM_AP_CFG_PRM_NC_DROP_BUDGET_VALID_FLAGS;

#define PNM_AP_CFG_NC_DROPPED_FRAMES_MAX          (0x03E7)      /**< maximum number of allowed dropped frames */

#define PNM_AP_CFG_PRM_PDNCDATACHECK_VERSION_1   (0x0001)

typedef struct PNM_AP_CFG_PRM_PDNCDATACHECK_Ttag PNM_AP_CFG_PRM_PDNCDATACHECK_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDNCDATACHECK_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;
  /** field containing the valid parameters of this packet - see above */
  uint32_t ulEnableFlag;
  /** Max number of dropped frames that accepted per second before generating
   *  a Maintenance Demanded message
      \par See Spec:
       Coding of the field NCDropBudgetType
  */
  uint32_t ulMaintenanceRequiredDropBudget;
  /** Max number of dropped frames that accepted per second before generating
   *  a Maintenance Required message
      \par See Spec:
       Coding of the field NCDropBudgetType
  */
  uint32_t ulMaintenanceDemandedDropBudget;
  /** Max number of dropped frames that accepted per second before generating
   *  a Diagnosis message
      \par See Spec:
       Coding of the field NCDropBudgetType
  */
  uint32_t ulErrorDropBudget;
};

#define PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_TIMEIOINPUT_MAX       (32000000)
#define PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_TIMEIOOUTPUT_MAX      (32000000)
#define PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_TIMEIOINPUTVALID_MAX  (32000000)
#define PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_TIMEIOOUTPUTVALID_MAX (32000000)

#define PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_CONTROLLERAPPLICATIONCYCLEFACTOR_MIN (0x0001)
#define PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_CONTROLLERAPPLICATIONCYCLEFACTOR_MAX (0x0400)

#define PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_TIMEDATACYCLE_MIN     (0x0001) /* 31,25 microseconds */
#define PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_TIMEDATACYCLE_MAX     (0x0400) /* 32 milliseconds */

#define PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_VERSION_1             (0x0001)

typedef struct PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_Ttag PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;
  /** cycle time of controller application in irt bus cycles (send clock) */
  uint16_t usControllerApplicationCycleFactor;
  /** Time of on datacycle in units of 31.25 �s */
  uint16_t usTimeDataCycle;
  /** Timepoint when to latch the input data in nanoseconds to next application start */
  uint32_t ulTimeIOInput;       /* in nanoseconds */
  /** Timepoint when to set the output data in nanoseconds since last application start */
  uint32_t ulTimeIOOutput;      /* in nanoseconds */
  uint32_t ulTimeIOInputValid;  /* in nanoseconds */
  uint32_t ulTimeIOOutputValid; /* in nanoseconds */
};

#define PNM_AP_CFG_PRM_ISOCHRONOUSCONTROLLERDATA_VERSION_1             (0x0001)

typedef struct PNM_AP_CFG_PRM_ISOCHRONOUSCONTROLLERDATA_Ttag PNM_AP_CFG_PRM_ISOCHRONOUSCONTROLLERDATA_T;

/** Isochronous application data for controller.
 *
 * The following structure is not a real record but may be used in
 * future to pass isochronous application information from engineering
 * to controller application.
 */
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_ISOCHRONOUSCONTROLLERDATA_Ttag
{
  /** Version of the structure */
  uint32_t ulStructVersion;
  /** Time required by the profinet controller firmware to transfer the input data
   * from internal buffers to DPM memory in nanoseconds
   */
  uint32_t ulTimeControllerApplicationInputMin;
  /** Time required by the profinet controller firmware to transfer the output data
   * from DPM memory to internal buffers in nanoseconds.
   */
  uint32_t ulTimeControllerApplicationOutputMin;
  /** Time when isochronous mode data is available in internal buffers in profinet
   * controller firmware. This time mainly depends on the topology and configuration.
   * Relative to start of bus cycle in nanoseconds
   */
  uint32_t ulTimeControllerApplicationValid;
  /** Calculated start time of isochronous controller application
   * relative to start of bus cycle. Specified in units of nanoseconds
   */
  uint32_t ulTimeControllerApplicationStart;
  /** Calculated end time of isochronous controller application
   * relative to start of bus cycle. Specified in units of nanoseconds.
   *
   * The application processing must be finished until this time in order
   * to send the data in the next bus cycle. This includes writing and handshaking the
   * DPM output area.
   */
  uint32_t ulTimeControllerApplicationEnd;
};

typedef enum
{
  /** Manufacturer specific record. Use in Conjunction
   * with Record Data for IO Device */
  PNM_AP_CFG_PRM_RECORD                     =  0,
  /** Specific records defined in Profinet specification to follow */
  PNM_AP_CFG_PRM_PDPORTDATACHECK            =  1,
  PNM_AP_CFG_PRM_PDPORTDATAADJUST           =  2,
  PNM_AP_CFG_PRM_PDIRDATA_PDIRGLOBALDATA    =  3,
  PNM_AP_CFG_PRM_PDIRDATA_PDIRFRAMEDATA     =  4,
  PNM_AP_CFG_PRM_PDIRDATA_PDIRBEGINENDDATA  =  5,
  PNM_AP_CFG_PRM_PDSYNCDATA                 =  6,
  PNM_AP_CFG_PRM_PDINTFMRPDATACHECK         =  7,
  PNM_AP_CFG_PRM_PDINTFMRPDATAADJUST        =  8,
  PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST        =  9,
  PNM_AP_CFG_PRM_PDPORTFODATACHECK          = 10,
  PNM_AP_CFG_PRM_PDPORTFODATAADJUST         = 11,
  PNM_AP_CFG_PRM_PDNCDATACHECK              = 12,
  PNM_AP_CFG_PRM_PDINTERFACEADJUST          = 13,
  PNM_AP_CFG_PRM_PDINTERFACEFSUDATAADJUST   = 14,
  PNM_AP_CFG_PRM_ARFSUDATAADJUST            = 15,
  PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA        = 16,
  PNM_AP_CFG_PRM_ISOCHRONOUSCONTROLLERDATA  = 17,
  PNM_AP_CFG_PRM_MAX,
} PNM_AP_CFG_PRM_TYPE_E;

/* parameter union */
typedef union PNM_AP_CFG_IOC_PRM_UNION_Ttag PNM_AP_CFG_IOC_PRM_UNION_T;
union PNM_AP_CFG_IOC_PRM_UNION_Ttag
{
  PNM_AP_CFG_PRM_PDINTERFACEADJUST_T         tIntfAdjust;
  PNM_AP_CFG_PRM_SYNC_T                      tSyncData;
  PNM_AP_CFG_PRM_IRDATA_GLOBAL_T             tIrtGlobal;
  PNM_AP_CFG_PRM_IRDATA_PHASES_T             tIrtPhases;
  PNM_AP_CFG_PRM_IRDATA_FRAME_T              tIrtFrames;
  PNM_AP_CFG_PRM_PDINTFMRPDATACHECK_T        tIntfMRPDataCheck;
  PNM_AP_CFG_PRM_PDINTFMRPDATAADJUST_T       tIntfMRPDataAdjust;
  PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_T       tPortMRPDataAdjust;
  PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_T         tPortDataAdjust;
  PNM_AP_CFG_PRM_PDPORTDATA_CHECK_T          tPortDataCheck;
  PNM_AP_CFG_PRM_PDNCDATACHECK_T             tNCDataCheck;
  PNM_AP_CFG_PRM_ISOCHRONOUSCONTROLLERDATA_T tIsochronousControllerData;
};

/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_IOC_PRM_STRUCT_VERSION_1    (0x0001)

/** request packet data */
typedef struct PNM_AP_CFG_IOC_PRM_DATA_Ttag PNM_AP_CFG_IOC_PRM_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_IOC_PRM_DATA_Ttag
{
  /** structure version of this structure */
  uint32_t                   ulStructVersion;

  /** identifier see \ref PNM_AP_CFG_PRM_TYPE_E */
  uint16_t                   usPrmType;
  /** Port specifier.
   *
   * @details Some records are related to a specific port of the controller.
   * Allowed values: 0 = Interface, 1 = Port 0, 2 = Port 1 */
  uint8_t                     bPortId;
  /** Reserved (Padding) */
  uint8_t                     bPadding;
  /** parameter data */
  PNM_AP_CFG_IOC_PRM_UNION_T  uData;
} ;

/** request packet */
typedef struct PNM_AP_CFG_IOC_PRM_REQ_Ttag PNM_AP_CFG_IOC_PRM_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_IOC_PRM_REQ_Ttag
{
  /** packet header */
  PNM_AP_PCK_HEADER_T                    tHead;
  /** packet data */
  PNM_AP_CFG_IOC_PRM_DATA_T              tData;
};

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T            PNM_AP_CFG_IOC_PRM_CNF_T;

/** packet union */
typedef union PNM_AP_CFG_IOC_PRM_PCK_Ttag PNM_AP_CFG_IOC_PRM_PCK_T;

union PNM_AP_CFG_IOC_PRM_PCK_Ttag
{
  /** request packet */
  PNM_AP_CFG_IOC_PRM_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_CFG_IOC_PRM_CNF_T    tCnf;
};



/*******************************************************************************
 CONFIG IO-DEVICE SERVICE
*******************************************************************************/

/** usable flags in ulFlags */
enum PNM_AP_CFG_IOD_FLAG_Etag
{
  /** the IO-Device supports WriteMultiple */
  PNM_AP_CFG_IOD_FLAG_WRITE_MULTIPLE_SUPPORTED = 0x01,
  /** IO Controller firmware shall not check parameters inside this request */
  PNM_AP_CFG_IOD_FLAG_DISABLE_PRM_CHECK        = 0x02,
  /** The AR is initially disabled. If this flag is set, the profinet controller firmware
   * will not establish this AR on startup */
  PNM_AP_CFG_IOD_FLAG_DISABLED                 = 0x04,
  /** This bit is reserved for System redundancy usage. Do not set */
  PNM_AP_CFG_IOD_FLAG_RESERVED_SR              = 0x08,

  PNM_AP_CFG_IOD_FLAG_MAX,
};

/** Device address modes */
enum PNM_AP_CFG_IOD_ADDRESS_MODE_Etag
{
  /** The controller identifies the device using its name of station
   * or alias name and will assign name of station and ip if required.
   *
   * This is the default operaiton mode as defined in specification */
  PNM_AP_CFG_IOD_ADDRESS_MODE_DCP_IDENTIFY_ASSIGN = 0x00,
  /** The controller discovers the device using the configured name of station
   * but does not configure the ipsuite. The controller will use the ip address
   * reported by the device to establish communication. The controller will
   * not establish communication if no ip address was preconfigured in the device
   */
  PNM_AP_CFG_IOD_ADDRESS_MODE_DCP_DISCOVER        = 0x01,
  /** The controller discovers the device using its ip address by means of arp
   * no dcp is sent
   */
  PNM_AP_CFG_IOD_ADDRESS_MODE_ARP_DISCOVER        = 0x02,
};

typedef enum PNM_AP_CFG_IOD_ADDRESS_MODE_Etag PNM_AP_CFG_IOD_ADDRESS_MODE_E;

typedef enum PNM_AP_CFG_IOD_FLAG_Etag PNM_AP_CFG_IOD_FLAG_E;

/* AR Types */
enum PNM_AP_CFG_IOD_AR_TYPE_Etag
{
  /** standard AR Type for RT communication */
  PNM_AP_CFG_IOD_AR_TYPE_SINGLE           = 0x01,
  /** IO Supervisor AR (with IO-data), not supported */
  PNM_AP_CFG_IOD_AR_TYPE_SUPERVISOR       = 0x06,
  /** standard AR Type for IRT communication */
  PNM_AP_CFG_IOD_AR_TYPE_RTC3             = 0x10,
  /** SystemRedundancy AR. Requires Structversion 2 */
  PNM_AP_CFG_IOD_AR_TYPE_SYSTEMREDUNDANCY = 0x20,
};

typedef enum PNM_AP_CFG_IOD_AR_TYPE_Etag PNM_AP_CFG_IOD_AR_TYPE_E;

/* AR Properties */
enum PNM_AP_CFG_IOD_AR_PROP_FLAGS_Etag
{
  /* a supervisor is allowed to take over submodules of this AR */
  PNM_AP_CFG_IOD_AR_PROP_FLAG_SUPERVISOR_TAKEOVER_ALLOWED     = 0x00000008,
  /** startup according to Profinet specification 2.3, if not set the legacy startup is used */
  PNM_AP_CFG_IOD_AR_PROP_FLAG_STARTUPMODE_ADVANCED            = 0x40000000,
  /* alarm type PULL MODULE is supported */
  PNM_AP_CFG_IOD_AR_PROP_FLAG_PULL_MODULE_ALARM_ALLOWED       = 0x80000000,
  /* AR is an device access AR, only valid in conjunction with
   * PNM_AP_CFG_IOD_AR_TYPE_SUPERVISOR */
  PNM_AP_CFG_IOD_AR_PROP_FLAG_DEVICEACCESS                    = 0x00000100,
};

typedef enum PNM_AP_CFG_IOD_AR_PROP_FLAGS_Etag PNM_AP_CFG_IOD_AR_PROP_FLAGS_E;

typedef uint16_t PNM_AP_DEVICEHANDLE_T;

enum PNM_AP_DEVICEHANDLE_Etag
{
  /** Device handle of the controller itself */
  PNM_AP_DEVICEHANDLE_IOC        = 0x0000,

  /** First valid devicehandle to use for IO AR */
  PNM_AP_DEVICEHANDLE_IOAR_FIRST = 0x0001,
  /** Last valid devicehandle to use for IO AR */
  PNM_AP_DEVICEHANDLE_IOAR_LAST  = 0x0100,

  /** First valid devicehandle to use for DA AR */
  PNM_AP_DEVICEHANDLE_DAAR_FIRST = 0xF000,
  /** Last valid devicehandle to use for DA AR */
  PNM_AP_DEVICEHANDLE_DAAR_LAST  = 0xF000,
};

/* AR Properties */
enum PNM_AP_CFG_IOD_AR_SRPROP_FLAGS_Etag
{
  /** When this flag is set, the device will send valid input data even if AR is in
   * Backup state. This feature must not be enabled if not supported by the device
   * (GSDML file) */
  PNM_AP_CFG_IOD_AR_SRPROP_FLAG_INPUTVALIDONBACKUPAR = 0x00000001,
  /** These bits are managed by controller firmware internally. Application can not configure
   * this setting. Application shall never set any of these bits. */
  PNM_AP_CFG_IOD_AR_SRPROP_FLAG_RESERVED_MASK        = 0x00000006,
};


/* RTA Retries */
#define PNM_AP_CFG_IOD_RTA_RETRIES_MIN            (0x0003)  /**< minimum value allowed for RTA retries */
#define PNM_AP_CFG_IOD_RTA_RETRIES_MAX            (0x000F)  /**< maximum value allowed for RTA retries */

/* RTA timeout factor */
#define PNM_AP_CFG_IOC_RTA_TIMEOUT_FACT_MIN       (0x0001)
#define PNM_AP_CFG_IOC_RTA_TIMEOUT_FACT_MAX       (0x0064)

/* Redundancy Data Hold Factor */
#define PNM_AP_CFG_IOC_RDHT_FACT_MIN              (0x0003)
#define PNM_AP_CFG_IOC_RDHT_FACT_MAX              (0xFFFF)

/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_IOD_STRUCT_VERSION_1    (0x0001)
#define PNM_AP_CFG_IOD_STRUCT_VERSION_2    (0x0002)

/** request packet data */
typedef struct PNM_AP_CFG_IOD_REQ_DATA_Ttag PNM_AP_CFG_IOD_REQ_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_IOD_REQ_DATA_Ttag
{
  /** structure version of this structure */
  uint32_t              ulStructVersion;
  /** unique handle of this IO-Device defined by the sender of this packet */
  PNM_AP_DEVICEHANDLE_T usDeviceHandle;
  /** AR Type see \ref PNM_AP_CFG_IOD_AR_TYPE_E */
  uint8_t               bARType;
  /** Address mode see \ref PNM_AP_CFG_IOD_ADDRESS_MODE_E */
  uint8_t               bAddressMode;
  /** flags see \ref PNM_AP_CFG_IOD_FLAG_E */
  uint32_t              ulFlags;
  /** AR UUID - has to be unique over all configured IO-Devices
      \par See Spec:
       Coding of the field ARUUID
  */
  PNM_UUID_T            tArUuid;
  /** AR Properties see \ref PNM_AP_CFG_IOD_AR_PROP_FLAGS_E
      \par See Spec:
       Coding of the field ARProperties
  */
  uint32_t              ulArProperties;
  /** VendorID */
  uint16_t              usVendorID;
  /** DeviceID */
  uint16_t              usDeviceID;
  /** InstanceID (GSDML parameter ObjectUUID Instance) */
  uint16_t              usInstanceID;
  /** max. alarm data length */
  uint16_t              usMaxAlarmDataLength;
  /* RTA Timeout Factor
     \par See Spec:
      Coding of the field RTATimeoutFactor
  */
  uint16_t              usRTATimeoutFact;
  /** RTA Retries
      \par See Spec:
       Coding of the field RTARetries
  */
  uint16_t              usRTARetries;
  /** NameOfStation of the IO Device
      zero padded: length determined by searching first zero
      \note  The GSDML Element \c OrderNumber.
      \par See Spec:
       Coding of the field NameOfStationValue
  */
  uint8_t               abNameOfStation[240];
  /** IP address */
  uint32_t              ulIPAddr;
  /** network mask */
  uint32_t              ulNetworkMask;
  /** Gateway address */
  uint32_t              ulGatewayAddr;

  /* next fields are structversion 2 and above only */

  /** System Redundancy Properties for this AR. Only evaluated if
   * bARType == PNM_AP_CFG_IOD_AR_TYPE_SYSTEMREDUNDANCY. */
  uint32_t              ulSRProperties;
  /** Redundancy data hold factor is the redundancy data hold timeout
   * in units of 1ms. Only evaluated if
   * bARType == PNM_AP_CFG_IOD_AR_TYPE_SYSTEMREDUNDANCY. */
  uint16_t              usRDHTFactor;
  /** Redundancy MTOT short is the maximal switchover time in units of 1ms. */
  uint32_t              ulMtotShort;
};

/** request packet */
typedef struct PNM_AP_CFG_IOD_REQ_Ttag PNM_AP_CFG_IOD_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_IOD_REQ_Ttag
{
  /** packet header */
  PNM_AP_PCK_HEADER_T              tHead;
  /** packet data */
  PNM_AP_CFG_IOD_REQ_DATA_T        tData;
};

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T         PNM_AP_CFG_IOD_CNF_T;

/** packet union */
typedef union PNM_AP_CFG_IOD_PCK_Ttag PNM_AP_CFG_IOD_PCK_T;
union PNM_AP_CFG_IOD_PCK_Ttag
{
  /** request packet */
  PNM_AP_CFG_IOD_REQ_T          tReq;
  /** confirmation packet */
  PNM_AP_CFG_IOD_CNF_T          tCnf;
};


/*******************************************************************************
 CONFIGURE AR PARAMETERS SERVICE
*******************************************************************************/

typedef uint16_t PNM_AP_ARVENDORBLOCKHANDLE_T;

enum PNM_AP_ARVENDORBLOCKHANDLE_Etag
{
  /** The first handle value to use for AR Vendorblocks*/
  PNM_AP_ARVENDORBLOCKHANDLE_FIRST = 0x0001,
  /** The last handle value to use for AR Vendorblocks*/
  PNM_AP_ARVENDORBLOCKHANDLE_LAST  = 0x0100,
};

/* AR Vendor Block, refer to GSDML ARVendorBlock */
/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_AR_PRM_ARVENDORBLOCK_STRUCT_VERSION_1    (0x0001)

typedef struct PNM_AP_CFG_AR_PRM_ARVENDORBLOCK_Ttag PNM_AP_CFG_AR_PRM_ARVENDORBLOCK_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_AR_PRM_ARVENDORBLOCK_Ttag
{
  /** structure version of this structure */
  uint32_t                      ulStructVersion;
  /** handle of this AR VendorBlock, required to read out the Response at runtime */
  PNM_AP_ARVENDORBLOCKHANDLE_T  usArVendorBlockHandle;
  /** AP Structure Identifier according to GSDML */
  uint16_t                      usAPStructureIdentifier;
  /** API according to GSDML */
  uint32_t                      ulApi;
  /** Array of the VendorBlock data, sizeof array depend on parameter usDataLen  */
  uint8_t                       abData[ARRAYS_OF_LENGTH_ZERO];
} ;

/* IR Data */
/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_AR_PRM_IRDATA_STRUCT_VERSION_1    (0x0001)

typedef struct PNM_AP_CFG_AR_PRM_IRDATA_Ttag PNM_AP_CFG_AR_PRM_IRDATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_AR_PRM_IRDATA_Ttag
{
  /** structure version of this structure */
  uint32_t              ulStructVersion;
  /* RTC3 domain UUID of this IRT IO device
     needs to match the UUID of IO Controllers Sync parameters
     needs to match the UUID of IO Devices PDSyncData parameters
     \par See Spec:
      Coding of the field IRDataUUID
  */
  PNM_UUID_T            tIrDataUuid;
};

#ifdef PNM_SUPPORT_PARAMETER_SERVER
#warning "usage of Parameter Server is not supported..."
/* Parameter Server */
/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_AR_PRM_PARAMETERSERVER_BLOCK_STRUCT_VERSION_1    (0x0001)

typedef struct PNM_AP_CFG_AR_PRM_PARAMETERSERVER_BLOCK_Ttag PNM_AP_CFG_AR_PRM_PARAMETERSERVER_BLOCK_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_AR_PRM_PARAMETERSERVER_BLOCK_Ttag
{
  /* external Parameter server ObjUUID */
  PNM_UUID_T            tPrmServerUuid;
  uint32_t              ulPrmServerProp;
  uint32_t              ulPrmServerActivityTimeoutFactor;
  /** NameOfStation, zero padded: length determined by searching
   * first zero */
  uint8_t               abPrmServerNameOfStation[240];
};
#endif

/* FastStartUp */
/* allowed values for field ulFSParamMode */
enum PNM_AP_CFG_AR_PRM_FSPARAM_MODE_Etag
{
  PNM_AP_CFG_AR_PRM_FSPARAM_MODE_DISABLE = 1,
  PNM_AP_CFG_AR_PRM_FSPARAM_MODE_ENABLE  = 2,
};
typedef enum PNM_AP_CFG_AR_PRM_FSPARAM_MODE_Etag PNM_AP_CFG_AR_PRM_FSPARAM_MODE_E;

/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_AR_PRM_FSU_PARAMETERS_STRUCT_VERSION_1    (0x0001)

typedef struct PNM_AP_CFG_PRM_AR_FSU_PARAMETERS_Ttag PNM_AP_CFG_PRM_AR_FSU_PARAMETERS_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_AR_FSU_PARAMETERS_Ttag
{
  /** structure version of this structure */
  uint32_t              ulStructVersion;
  /** FSParameterMode see \ref PNM_AP_CFG_AR_PRM_FSPARAM_MODE_E
      \par See Spec:
       Coding of the field FSParameterMode
  */
  uint32_t              ulFSParamMode;
  /** FSParameter UUID, irrelevant if FSParamMode is set to off
      \par See Spec:
       Coding of the field FSParameterUUID
  */
  PNM_UUID_T            tFSParamUuid;
};

/* identifier */
typedef enum
{
  PNM_AP_CFG_AR_PRM_IDENT_ARVENDORBLOCKREQ = 0,
  PNM_AP_CFG_AR_PRM_IDENT_IRDATA           = 1,
  PNM_AP_CFG_AR_PRM_IDENT_PARAMETERSERVER  = 2,
} PNM_AP_CFG_AR_PRM_IDENTIFIER_E;

typedef union
{
  PNM_AP_CFG_AR_PRM_ARVENDORBLOCK_T          tArVendorBlock;
  PNM_AP_CFG_AR_PRM_IRDATA_T                 tIrData;
#ifdef PNM_SUPPORT_PARAMETER_SERVER
  PNM_AP_CFG_AR_PRM_PARAMETERSERVER_BLOCK_T  tPrmServerBlock;
#endif
} PNM_AP_CFG_AR_PRM_UNION_T;

/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_AR_PRM_STRUCT_VERSION_1    (0x0001)

typedef struct PNM_AP_CFG_AR_PRM_DATA_Ttag PNM_AP_CFG_AR_PRM_DATA_T;
/** request packet data */
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_AR_PRM_DATA_Ttag
{
  /** structure version of this structure */
  uint32_t                ulStructVersion;
  /** IO-Device handle the submodule belongs to */
  PNM_AP_DEVICEHANDLE_T   usDeviceHandle;
  /** identifier see \ref PNM_AP_CFG_AR_PRM_IDENTIFIER_E */
  uint16_t                usIdentifier;
  /** parameter data */
  PNM_AP_CFG_AR_PRM_UNION_T uData;
};

typedef struct PNM_AP_CFG_AR_PRM_REQ_Ttag PNM_AP_CFG_AR_PRM_REQ_T;

/** request packet */
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_AR_PRM_REQ_Ttag
{
  /** packet header */
  PNM_AP_PCK_HEADER_T                tHead;
  /** packet data */
  PNM_AP_CFG_AR_PRM_DATA_T           tData;
};

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T            PNM_AP_CFG_AR_PRM_CNF_T;

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_AR_PRM_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_CFG_AR_PRM_CNF_T    tCnf;
} PNM_AP_CFG_AR_PRM_PCK_T;


/*******************************************************************************
 CONFIGURE IOCR SERVICE
*******************************************************************************/
typedef uint16_t PNM_AP_IOCR_HANDLE_T;

enum PNM_AP_IOCR_HANDLE_Etag
{
  /** minimum value for IOCR handle with type Input */
  PNM_AP_IOCR_HANDLE_INPUT_FIRST = 0x1000,
  /** maximum value for IOCR handle with type Input */
  PNM_AP_IOCR_HANDLE_INPUT_LAST  = 0x107F,

  /** minimum value for IOCR handle with type Output */
  PNM_AP_IOCR_HANDLE_OUTPUT_FIRST = 0x2000,
  /** maximum value for IOCR handle with type Output */
  PNM_AP_IOCR_HANDLE_OUTPUT_LAST  = 0x207F,
};

typedef enum PNM_AP_IOCR_HANDLE_Etag PNM_AP_IOCR_HANDLE_E;

/* IOCR Type */
typedef enum
{
  /* this is an Input CR */
  PNM_AP_CFG_IOCR_TYPE_INPUT              = 1,
  /* this is an Output CR */
  PNM_AP_CFG_IOCR_TYPE_OUTPUT             = 2,
  /* not supported, for future use */
  PNM_AP_CFG_IOCR_TYPE_MULTICAST_PROVIDER = 3,
  /* not supported, for future use */
  PNM_AP_CFG_IOCR_TYPE_MULTICAST_CONSUMER = 4,
  /* invalid value */
  PNM_AP_CFG_IOCR_TYPE_MAX,
} PNM_AP_CFG_IOCR_TYPE_E;

/* IOCR Properties */
typedef enum
{
  PNM_AP_CFG_IOCR_PROP_RT_CLASS_1_LEGACY  = 0x01,    /* RT_CLASS_1, use legacy FrameIDs (0xC000-0xF7FF) */
  PNM_AP_CFG_IOCR_PROP_RT_CLASS_1         = 0x02,    /* RT_CLASS_1, use FrameIDs (0x8000-0xBBFF) */
  PNM_AP_CFG_IOCR_PROP_RT_CLASS_3         = 0x03,    /* RT_CLASS_3 (IRT), use FrameIDs (0x0100-0x06FF) */
  PNM_AP_CFG_IOCR_PROP_RT_CLASS_UDP       = 0x04,    /* not supported, for future use */
} PNM_AP_CFG_IOCR_PROP_E;

/** Definition of the frame ids the controller will use
 *
 * The following array defines the base value for frameid assignment
 * within the controller firmware. The frame id of a given IOCR will
 * be the IOCR Handle minus PNM_AP_IOCR_HANDLE_XXX_FIRST plus
 * the base value selected according IOCR properties */
enum PNM_AP_CFG_IOCR_FRAMEIDBASE_Etag
{
  /** Base value for RTC3 Input IOCR FrameIds
   *
   * The PnC XC Code will only accept RTC3 CPM with 0x100 <= Frame Id < 0x180.
   * This is hardcoded within XC !
   */
  PNM_AP_CFG_IOCR_FRAMEIDBASE_RTC3INPUT   = 0x0100,
  PNM_AP_CFG_IOCR_FRAMEIDBASE_RTC3OUTPUT  = 0x0200,
  PNM_AP_CFG_IOCR_FRAMEIDBASE_RTC1        = 0x8000,
  PNM_AP_CFG_IOCR_FRAMEIDBASE_RTC1LEGACY  = 0xC000,
};

typedef enum PNM_AP_CFG_IOCR_FRAMEIDBASE_Etag PNM_AP_CFG_IOCR_FRAMEIDBASE_E;

/* SendClockFactor */
#define PNM_AP_CFG_IOCR_SEND_CLOCK_DEFAULT         (0x0020)  /**< default value for SendClockFactor */
#define PNM_AP_CFG_IOCR_SEND_CLOCK_MIN_RTC1        (0x0008)  /**< minimum value allowed for SendClockFactor for RT_CLASS_1 */
#define PNM_AP_CFG_IOCR_SEND_CLOCK_MIN_RTC3        (0x0008)  /**< minimum value allowed for SendClockFactor for RT_CLASS_3 */
#define PNM_AP_CFG_IOCR_SEND_CLOCK_32              (0x0020)  /**< minimum value allowed for SendClockFactor for RT_CLASS_1 */
#define PNM_AP_CFG_IOCR_SEND_CLOCK_64              (0x0040)  /**< minimum value allowed for SendClockFactor for RT_CLASS_1 */
#define PNM_AP_CFG_IOCR_SEND_CLOCK_MAX             (0x0080)  /**< maximum value allowed for SendClockFactor */

/* ReductionRatio */
/* the possible values depend on SendClock */
#define PNM_AP_CFG_IOCR_REDUCT_RATIO_MIN           (0x0001)  /**< minimum value allowed for ReductionRatio */
#define PNM_AP_CFG_IOCR_REDUCT_RATIO_256           (0x0100)  /**< maximum value allowed for ReductionRatio for RT_CLASS_1 */
#define PNM_AP_CFG_IOCR_REDUCT_RATIO_MAX_RTC1      (0x0200)  /**< maximum value allowed for ReductionRatio for RT_CLASS_1 */
#define PNM_AP_CFG_IOCR_REDUCT_RATIO_MAX_RTC3      (0x0010)  /**< maximum value allowed for ReductionRatio for RT_CLASS_3 */

/* Phase */
/* the possible values depend on SendClock and ReductionRatio */
#define PNM_AP_CFG_IOCR_PHASE_VALUE_MIN            (0x0001)  /**< minimum value allowed for Phase */
#define PNM_AP_CFG_IOCR_PHASE_VALUE_MAX            (0x4000)  /**< maximum value allowed for Phase */

/* FrameSendOffset */
#define PNM_AP_CFG_IOCR_FRAME_SEND_OFFSET_RTC1     (0xFFFFFFFF) /**< best practice, default value */

/* DataHold Factor */
#define PNM_AP_CFG_IOCR_DATAHOLD_DEFAULT           (0x0003)    /**< default value for datahold factor */
#define PNM_AP_CFG_IOCR_DATAHOLD_MIN               (0x0001)    /**< minimum value allowed for datahold factor */
#define PNM_AP_CFG_IOCR_DATAHOLD_MAX               (0x1E00)    /**< maximum value allowed for datahold factor */

/** Maximum value allowed for dataholdTime of 1,92 s in base of 31,25us (equal to 1920000us/31.25us) */
#define PNM_AP_CFG_IOCR_DATAHOLD_TIME_MAX          (61440)


/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_IOCR_STRUCT_VERSION_1    (0x0001)

typedef struct PNM_AP_CFG_IOCR_DATA_Ttag PNM_AP_CFG_IOCR_DATA_T;

/** request packet data */
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_IOCR_DATA_Ttag
{
  /** structure version of this structure */
  uint32_t                ulStructVersion;
  /** this IOCRs unique handle */
  PNM_AP_IOCR_HANDLE_T    usIocrHandle;
  /** IO-Device handle the IOCR belongs to */
  PNM_AP_DEVICEHANDLE_T   usDeviceHandle;

  /** Flags for future use, set to 0 */
  uint32_t                ulFlags;
  /** IOCR properties see \ref PNM_AP_CFG_IOCR_PROP_FLAG_E
      \par See Spec:
       Coding of the field IOCRProperties
  */
  uint32_t                ulIocrProp;
  /** IOCR type see \ref PNM_AP_CFG_IOCR_TYPE_E
      \par See Spec:
       Coding of the field IOCRType
  */
  uint16_t                usIocrType;
  /** IOCR multicast MAC address for future use, currently unsupported, set to 0
      \par See Spec:
       Coding of the field IOCRMulticastMACAdd
  */
  uint8_t                 abMcastMACAddr[6];
  /** IOCR data length, the length of the cyclic frame represented by this IOCR
      \par See Spec:
       Coding of the field DataLength
  */
  uint16_t                usDataLen;
  /** IOCR sendclock factor - see above
      \par See Spec:
       Coding of the field SendClockFactor
  */
  uint16_t                usSendClockFact;
  /** IOCR reduction ratio - see above
      \par See Spec:
       Coding of the field ReductionRatio
  */
  uint16_t                usReductRatio;
  /** IOCR phase - see above
      \par See Spec:
       Coding of the field Phase
  */
  uint16_t                usPhase;
  /** IOCR sequence number
      \par See Spec:
       Coding of the field Sequence
  */
  uint16_t                usSequence;
  /** IOCR datahold factor - see above
      \par See Spec:
       Coding of the field DataHoldFactor
  */
  uint16_t                usDataHoldFact;
  /** IOCR FrameSendOffset - see above
      \par See Spec:
       Coding of the field FrameSendOffset
  */
  uint32_t                ulFrameSendOffs;

  /** base address for the whole IO data of this IOCR in IO Controllers DPM input/output area */
  uint16_t                usDpmOffset;

  /** TODO: support vlan tag setting? */
};

typedef struct PNM_AP_CFG_IOCR_REQ_Ttag PNM_AP_CFG_IOCR_REQ_T;
/** request packet */
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_IOCR_REQ_Ttag
{
  /** packet header */
  PNM_AP_PCK_HEADER_T          tHead;
  /** packet data */
  PNM_AP_CFG_IOCR_DATA_T       tData;
};

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T    PNM_AP_CFG_IOCR_CNF_T;

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_IOCR_REQ_T          tReq;
  /** confirmation packet */
  PNM_AP_CFG_IOCR_CNF_T          tCnf;
} PNM_AP_CFG_IOCR_PCK_T;


/*******************************************************************************
 CONFIGURE SUBMODULE SERVICE
*******************************************************************************/
typedef uint16_t PNM_AP_SUBMODULE_HANDLE_T;


/** Submodule Properties Bitmask */
typedef enum
{
  PNM_AP_CFG_SUBMODULE_PROPERTIES_INPUT                        = 0x0001, /* Set If Input Data Available */
  PNM_AP_CFG_SUBMODULE_PROPERTIES_OUTPUT                       = 0x0002, /* Set If Output Daata Available*/

  PNM_AP_CFG_SUBMODULE_PROPERTIES_ACCESS_FULL                  = 0x0000, /* full access to submodule (default)*/
  PNM_AP_CFG_SUBMODULE_PROPERTIES_ACCESS_SHARED                = 0x0004, /* Shared Input submodule, no alarms and no parameterization */

  PNM_AP_CFG_SUBMODULE_PROPERTIES_ZERO_INPUT_DATA_LENGTH       = 0x0008, /* not supported, for future use */
  PNM_AP_CFG_SUBMODULE_PROPERTIES_ZERO_OUTPUT_DATA_LENGTH      = 0x0010, /* not supported, for future use */

  PNM_AP_CFG_SUBMODULE_PROPERTIES_USE_IOXS                     = 0x0000, /* IOXS enabled, (default) */
  PNM_AP_CFG_SUBMODULE_PROPERTIES_DISCARD_IOXS                 = 0x0020, /* not supported, for future use */
} PNM_AP_CFG_SUBMODULE_PROPERTIES_E;

/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_SUBMODULE_STRUCT_VERSION_1    (0x0001)

typedef struct PNM_AP_CFG_SUBMODULE_DATA_Ttag PNM_AP_CFG_SUBMODULE_DATA_T;
/** request packet data */
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_SUBMODULE_DATA_Ttag
{
  /** structure version of this structure */
  uint32_t                  ulStructVersion;
  /** unique submodule handle (freely defined by the user) */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
  /** IO-Device handle the submodule belongs to */
  PNM_AP_DEVICEHANDLE_T     usDeviceHandle;
  /** consumer IOCR handle the submodule belongs to */
  PNM_AP_IOCR_HANDLE_T      usInputIocrHandle;
  /** provider IOCR handle the submodule belongs to */
  PNM_AP_IOCR_HANDLE_T      usOutputIocrHandle;
  /** module identifier according to GSDML */
  uint32_t                  ulModuleIdentNumber;
  /** submodule identifier according to GSDML */
  uint32_t                  ulSubmoduleIdentNumber;
  /** API the submodule belongs to */
  uint32_t                  ulApi;
  /** slot the submodule is expected in */
  uint16_t                  usSlot;
  /* subslot the submodule is expected in */
  uint16_t                  usSubslot;
  /** submodule flags see \ref PNM_AP_CFG_SUBMODULE_FLAGS_E */
  uint16_t                  usSubmoduleProperties;
  /** length of the data this submodule provides according to GSDML */
  uint16_t                  usDataLenInput;
  /** length of the data this submodule consumes according to GSDML */
  uint16_t                  usDataLenOutput;
  /** offset inside the frame where the input data of this submodule shall reside */
  uint16_t                  usFrameOffsetInput;
  /** offset inside the frame where the output data of this submodule shall reside */
  uint16_t                  usFrameOffsetOutput;
  /** offset inside the frame where IOCS of input data of this submodule shall reside.
      Note that this is an offset in the output IOCR */
  uint16_t                  usIOCSFrameOffsetInput;
  /** offset inside the frame where IOCS of output data of this submodule shall reside.
      Note that this is an offset in the input IOCR */
  uint16_t                  usIOCSFrameOffsetOutput;
};

typedef struct PNM_AP_CFG_SUBMODULE_REQ_Ttag PNM_AP_CFG_SUBMODULE_REQ_T;
/** request packet */
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_SUBMODULE_REQ_Ttag
{
  /** packet header */
  PNM_AP_PCK_HEADER_T          tHead;
  /** packet data */
  PNM_AP_CFG_SUBMODULE_DATA_T  tData;
};

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T     PNM_AP_CFG_SUBMODULE_CNF_T;

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_SUBMODULE_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_CFG_SUBMODULE_CNF_T    tCnf;
} PNM_AP_CFG_SUBMODULE_PCK_T;


/*******************************************************************************
 CONFIGURE RECORD SERVICE
*******************************************************************************/

/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_RECORD_STRUCT_VERSION_1    (0x0001)

typedef struct PNM_AP_CFG_RECORD_DATA_Ttag PNM_AP_CFG_RECORD_DATA_T;
/** request packet data */
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_RECORD_DATA_Ttag
{
  /** structure version of this structure */
  uint32_t                ulStructVersion;
  /* the index of the record - defined in GSDML */
  uint16_t                usIndex;
  /* the transfersequence value - defined in GSDML */
  uint16_t                usTransferSequence;
  /** Array of Record Data, length of the data is derived from packet length  */
  uint8_t                 abRecordData[ARRAYS_OF_LENGTH_ZERO];
};

#define PNM_AP_CFG_PRM_PDPORTFIBEROPTICDATACHECK_VERSION_1   (0x0001)

typedef struct PNM_AP_CFG_PRM_PDPORTFODATACHECK_Ttag PNM_AP_CFG_PRM_PDPORTFODATACHECK_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDPORTFODATACHECK_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;
  /** Min po
   *  a Diagnosis message
      \par See Spec:
       Coding of the field FiberOpticPowerBudgetType
  */
  uint32_t ulMaintenanceRequiredPowerBudget;
  /**  generating
   *  a Diagnosis message
      \par See Spec:
       Coding of the field FiberOpticPowerBudgetType
  */
  uint32_t ulMaintenanceDemandedPowerBudget;
  /** Error power budget specifies the minimum receive sensitivity that accepted
   *  a Diagnosis message
      \par See Spec:
       Coding of the field FiberOpticPowerBudgetType
  */
  uint32_t ulErrorPowerBudget;
};


/* values for fiber optic cable type */
typedef enum
{
  PNM_AP_CFG_FIBEROPTIC_CABLE_UNKNOWN                        = 0, /* No cable specified */
  PNM_AP_CFG_FIBEROPTIC_CABLE_FIXED_INSTALLATION             = 1, /* Inside or Outside cable with fixed installation */
  PNM_AP_CFG_FIBEROPTIC_CABLE_FLIXIBLE_INSTALLATION          = 2, /* Inside or Outside cable with flexible installation  */
  PNM_AP_CFG_FIBEROPTIC_CABLE_OUTDOOR_FIXED_INSTALLATION     = 3, /* Outdoor cable with fixed installation  */
} PNM_AP_CFG_FIBEROPTIC_CABLE_TYPE_E;

#define PNM_AP_CFG_PRM_PDPORTFIBEROPTICDATAADJUST_VERSION_1  (0x0001)

typedef struct PNM_AP_CFG_PRM_PDPORTFODATAADJUST_Ttag PNM_AP_CFG_PRM_PDPORTFODATAADJUST_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDPORTFODATAADJUST_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;
  /** This mode indicates the type of fiber optic media connected to the port.
      \par See Spec:
      Coding of the field FiberOpticType
  */
  uint32_t ulFiberOpticType;
  /** This mode indicates the type of fiber optic cable connected to the port (see above ) .
      \par See Spec:
      Coding of the field FiberOpticCableType
  */
  uint32_t ulFiberOpticCableType;
};


#define PNM_AP_CFG_PRM_PDINTFFSUDATAADJUST_VERSION_1   (0x0001)

enum PNM_AP_FS_HELLO_MODE_Etag
{
  PNM_AP_FS_HELLO_MODE_OFF          = 0,
  PNM_AP_FS_HELLO_MODE_LINKUP       = 1,
  PNM_AP_FS_HELLO_MODE_LINKUP_DELAY = 2,
};

enum PNM_AP_FS_HELLO_INTERVAL_Ttag
{
  PNM_AP_FS_HELLO_INTERVAL_30MS     = 30,
  PNM_AP_FS_HELLO_INTERVAL_50MS     = 50,
  PNM_AP_FS_HELLO_INTERVAL_100MS    = 100,
  PNM_AP_FS_HELLO_INTERVAL_300MS    = 300,
  PNM_AP_FS_HELLO_INTERVAL_500MS    = 500,
  PNM_AP_FS_HELLO_INTERVAL_1000MS   = 1000,
};

enum PNM_AP_FS_HELLO_RETRY_Ttag
{
  PNM_AP_FS_HELLO_RETRY_MIN         = 1,
  PNM_AP_FS_HELLO_RETRY_MAX         = 15,
};

enum PNM_AP_FS_HELLO_DELAY_Ttag
{
  PNM_AP_FS_HELLO_DELAY_OFF         = 0,
  PNM_AP_FS_HELLO_DELAY_50MS        = 50,
  PNM_AP_FS_HELLO_DELAY_100MS       = 100,
  PNM_AP_FS_HELLO_DELAY_500MS       = 500,
  PNM_AP_FS_HELLO_DELAY_1000MS      = 1000,
};

typedef struct PNM_AP_CFG_PRM_PDINTERFACEFSUDATAADJUST_Ttag PNM_AP_CFG_PRM_PDINTERFACEFSUDATAADJUST_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_PRM_PDINTERFACEFSUDATAADJUST_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;
  /** FS Hello Mode indicates when Hello DCP requests are sent (see PNM_AP_FS_HELLO_MODE_Etag).
      \par See Spec:
       Coding of the field FSParameterMode
  */
  uint32_t ulFSHelloMode;
  /** FS Hello Interval indicates the time between two successive Hello requests.
      \par See Spec:
      Coding of the field FSHelloInterval
  */
  uint32_t ulFSHelloInterval;
  /** FS Hello Retry indicates the number of retries for the Hello requests.
      \par See Spec:
       Coding of the field FSHelloRetry
  */
  uint32_t ulFSHelloRetry;
  /** FS Hello Mode Delay indicates the initial timeout before sending the first Hello request.
      \par See Spec:
       Coding of the field FSHelloDelay
  */
  uint32_t ulFSHelloDelay;
};


typedef union PNM_AP_CFG_PARAMETER_Ttag PNM_AP_CFG_PARAMETER_T;

/* supported Parameter settings */
union PNM_AP_CFG_PARAMETER_Ttag
{
  PNM_AP_CFG_PRM_PDINTERFACEADJUST_T                 tIntfAdjust;
  PNM_AP_CFG_PRM_PDINTERFACEFSUDATAADJUST_T          tIntfFSUDataAdjust;
  PNM_AP_CFG_PRM_SYNC_T                              tSyncData;
  PNM_AP_CFG_PRM_IRDATA_GLOBAL_T                     tIrtGlobal;
  PNM_AP_CFG_PRM_IRDATA_PHASES_T                     tIrtPhases;
  PNM_AP_CFG_PRM_IRDATA_FRAME_T                      tIrtFrames;
  PNM_AP_CFG_PRM_PDINTFMRPDATACHECK_T                tIntfMRPDataCheck;
  PNM_AP_CFG_PRM_PDINTFMRPDATAADJUST_T               tIntfMRPDataAdjust;
  PNM_AP_CFG_PRM_PDPORTDATA_ADJUST_T                 tPortDataAdjust;
  PNM_AP_CFG_PRM_PDPORTDATA_CHECK_T                  tPortDataCheck;
  PNM_AP_CFG_PRM_PDPORTFODATAADJUST_T                tPortFODataAdjust;
  PNM_AP_CFG_PRM_PDPORTFODATACHECK_T                 tPortFODataCheck;
  PNM_AP_CFG_PRM_PDNCDATACHECK_T                     tNCDataCheck;
  PNM_AP_CFG_PRM_AR_FSU_PARAMETERS_T                 tArFsuData;
  PNM_AP_CFG_PRM_PDPORTMRPDATAADJUST_T               tPortMRPDataAdjust;
  PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA_T               tIsochronousModeData;
};

typedef union PNM_AP_CFG_RECORD_UNION_DATA_Ttag PNM_AP_CFG_RECORD_UNION_DATA_T;

union PNM_AP_CFG_RECORD_UNION_DATA_Ttag
{
  PNM_AP_CFG_RECORD_DATA_T    tRecord;
  PNM_AP_CFG_PARAMETER_T      tParam;
};

typedef struct PNM_AP_CFG_RECORD_Ttag PNM_AP_CFG_RECORD_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_RECORD_Ttag
{
  /** structure version of this structure */
  uint32_t                       ulStructVersion;
  /** submodule handle the record belongs to for module specific records,
   * specify use one submodule of the module, for device specific records use one
   * submodule of the whole device, e.g. DAP submodule */
  PNM_AP_SUBMODULE_HANDLE_T      usSubmoduleHandle;

  /** identifier what data is inside this dataset,
   * see \ref PNM_AP_CFG_PRM_TYPE_E */
  uint16_t                       usPrmType;

  PNM_AP_CFG_RECORD_UNION_DATA_T uData;
};

/** request packet */
typedef  struct PNM_AP_CFG_RECORD_REQ_Ttag PNM_AP_CFG_RECORD_REQ_T;
__PACKED_PRE struct PNM_AP_CFG_RECORD_REQ_Ttag
{
  /** packet header */
  PNM_AP_PCK_HEADER_T                tHead;
  /** packet data */
  PNM_AP_CFG_RECORD_T                tData;
};

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T            PNM_AP_CFG_RECORD_CNF_T;

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_RECORD_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_CFG_RECORD_CNF_T    tCnf;
} PNM_AP_CFG_RECORD_PCK_T;


/*******************************************************************************
 TOPOLOGY INFORMATION SERVICE
*******************************************************************************/

/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_TOPO_STRUCT_VERSION_1    (0x0001)

/** request packet */
typedef struct PNM_AP_CFG_TOPO_DATA_Ttag PNM_AP_CFG_TOPO_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_TOPO_DATA_Ttag
{

  /** @note the following rules shall apply if this Info contains topology information about IO-Controller local Ports:
   *  - Device Handle shall be set to  0
   *  - Submodule handle shall be set to 0xFFF1 for Port 1 and to 0xFFF2 for Port 2
   *  */

  /** structure version of this structure */
  uint32_t                  ulStructVersion;
  /** IO-Device handle of 1. submodule to connect */
  PNM_AP_DEVICEHANDLE_T     usDeviceHandle1;
  /** submodule handle of the 1. IO-Device to be connected */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle1;
  /** IO-Device handle of 2. submodule to connect */
  PNM_AP_DEVICEHANDLE_T     usDeviceHandle2;
  /** submodule handle of the 2. IO-Device to be connected */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle2;
};

typedef struct PNM_AP_CFG_TOPO_REQ_Ttag PNM_AP_CFG_TOPO_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_TOPO_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T         tHead;
  PNM_AP_CFG_TOPO_DATA_T tData;
};

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T            PNM_AP_CFG_TOPO_CNF_T;

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_TOPO_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_CFG_TOPO_CNF_T    tCnf;
} PNM_AP_CFG_TOPO_INFO_PCK_T;


/*******************************************************************************
 DOWNLOAD FINISHED SERVICE
*******************************************************************************/

/** request packet */
typedef PNM_AP_EMTPY_PCK_T   PNM_AP_DWNL_FIN_REQ_T;

/** confirmation packet */
typedef struct PNM_AP_DWNL_FIN_CNF_DATA_Ttag   PNM_AP_DWNL_FIN_CNF_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DWNL_FIN_CNF_DATA_Ttag
{

  /** The following attributes are only valid if an error occurs during configuration consistency checks  */
  /** handle of the affected IO-Device */
  PNM_AP_DEVICEHANDLE_T     usDeviceHandle;
  /** handle of the affected IOCR */
  PNM_AP_IOCR_HANDLE_T      usIocrHandle;
  /** handle of the affected Submodule */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
};

/** confirmation packet */
typedef struct PNM_AP_DWNL_FIN_CNF_Ttag   PNM_AP_DWNL_FIN_CNF_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_DWNL_FIN_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T         tHead;
  PNM_AP_DWNL_FIN_CNF_DATA_T  tData;
};


/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_DWNL_FIN_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_DWNL_FIN_CNF_T    tCnf;
} PNM_AP_DWNL_FIN_PCK_T;


/*******************************************************************************
 GET NUMBER OF CONFIGURED OBJECTS
*******************************************************************************/

/*
PNM_AP_CMD_CFG_GET_NUMBER_CONFIGURED_OBJECTS_REQ
    shall return numbers of configured
        ARs
        IOCRs
        submodules
    shall return the biggest values for
        DeviceHandle
        IOCRHandle (Input and Output)
        SubmoduleHandle
*/

/** request packet */
typedef PNM_AP_EMTPY_PCK_T PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_REQ_T;

typedef struct PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_CNF_DATA_Ttag PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_CNF_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_CNF_DATA_Ttag
{

  /** number of configured ARs */
  uint16_t usNumAR;
  /** number of configured IOCRs */
  uint16_t usNumIocr;
  /** number of configured Submodules */
  uint16_t usNumSubmodule;
  /* TBD: add biggest handle here as well? -> Ask AndreasMe */
};

/** confirmation packet */
typedef struct PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_CNF_Ttag PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_CNF_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T                              tHead;
  PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_CNF_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_REQ_T tReq;
  /** confirmation packet */
  PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_CNF_T tCnf;
} PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_PCK_T;

/*******************************************************************************
*******************************************************************************/

/*
PNM_AP_CMD_CFG_GET_NUM_OF_RECORDS_REQ
    shall return number of records for a specific submodule which is parameter for the request packet
    TODO
*/

/*******************************************************************************
 GET IO CONTROLLER PARAMETERS
*******************************************************************************/

/*
PNM_AP_CMD_CFG_GET_IOC_DATA_REQ
    return basic parameters of IO Controller itself
*/

#define PNM_AP_CFG_GET_IOC_STRUCT_VERSION_1    (0x0001)

/** request packet */
typedef struct PNM_AP_CFG_GET_IOC_REQ_DATA_Ttag PNM_AP_CFG_GET_IOC_REQ_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOC_REQ_DATA_Ttag
{
  /** structure version of this structure */
  uint32_t              ulStructVersion;
};

typedef struct PNM_AP_CFG_GET_IOC_REQ_Ttag PNM_AP_CFG_GET_IOC_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOC_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T           tHead;
  PNM_AP_CFG_GET_IOC_REQ_DATA_T tData;
};

/** confirmation packet */
typedef struct PNM_AP_CFG_GET_IOC_CNF_Ttag PNM_AP_CFG_GET_IOC_CNF_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOC_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T   tHead;
  PNM_AP_CFG_IOC_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_GET_IOC_REQ_T tReq;
  /** confirmation packet */
  PNM_AP_CFG_GET_IOC_CNF_T tCnf;
} PNM_AP_CFG_GET_IOC_PCK_T;

/*******************************************************************************
 GET IO PRM CONTROLLER PARAMETERS
*******************************************************************************/

#define PNM_AP_CFG_GET_IOC_PRM_LIST_STRUCT_VERSION_1 (0x0001)

/*
PNM_AP_CMD_CFG_GET_IOC_PRM_LIST_REQ
    return all "ParameterTypes" that have been configured for IO Controller
*/

/** request packet */
typedef PNM_AP_EMTPY_PCK_T PNM_AP_CFG_GET_IOC_PRM_LIST_REQ_T;

#define PNM_AP_MAX_IOC_PRM_TYPES 4

typedef struct PNM_AP_CFG_GET_IOC_PRM_LIST_CNF_DATA_Ttag PNM_AP_CFG_GET_IOC_PRM_LIST_CNF_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOC_PRM_LIST_CNF_DATA_Ttag
{
    uint32_t ulStructVersion;
    uint16_t uiPrmTypes[PNM_AP_MAX_IOC_PRM_TYPES];
    uint16_t uiPrmTypesSize;
};

/** confirmation packet */
typedef struct PNM_AP_CFG_GET_IOC_PRM_LIST_CNF_Ttag PNM_AP_CFG_GET_IOC_PRM_LIST_CNF_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOC_PRM_LIST_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T                    tHead;
  PNM_AP_CFG_GET_IOC_PRM_LIST_CNF_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_GET_IOC_PRM_LIST_REQ_T tReq;
  /** confirmation packet */
  PNM_AP_CFG_GET_IOC_PRM_LIST_CNF_T tCnf;
} PNM_AP_CFG_GET_IOC_PRM_LIST_PCK_T;

/*
PNM_AP_CMD_CFG_GET_IOC_PRM_REQ
    return the requested configuration for "ParameterType" and "Port Id"
*/
#define PNM_AP_CFG_GET_IOC_PRM_STRUCT_VERSION_1    (0x0001)


typedef struct PNM_AP_CFG_GET_IOC_PRM_REQ_DATA_Ttag PNM_AP_CFG_GET_IOC_PRM_REQ_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOC_PRM_REQ_DATA_Ttag
{
  uint32_t ulStructVersion;
  uint16_t uiPrmType;
  uint8_t  bPortId;
};

/** request packet */
typedef struct PNM_AP_CFG_GET_IOC_PRM_REQ_Ttag PNM_AP_CFG_GET_IOC_PRM_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOC_PRM_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T               tHead;
  PNM_AP_CFG_GET_IOC_PRM_REQ_DATA_T tData;
};

/** confirmation packet */
typedef struct PNM_AP_CFG_GET_IOC_PRM_CNF_Ttag PNM_AP_CFG_GET_IOC_PRM_CNF_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOC_PRM_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T       tHead;
  PNM_AP_CFG_IOC_PRM_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_GET_IOC_PRM_REQ_T tReq;
  /** confirmation packet */
  PNM_AP_CFG_GET_IOC_PRM_CNF_T tCnf;
} PNM_AP_CFG_GET_IOC_PRM_PCK_T;

/*******************************************************************************
 GET IO-DEVICE PARAMETERS
*******************************************************************************/

/*
PNM_AP_CMD_CFG_GET_IOD_REQ
    return basic parameters of an IO Device refered to by "DeviceHandle"
*/

typedef struct PNM_AP_CFG_GET_IOD_REQ_DATA_Ttag PNM_AP_CFG_GET_IOD_REQ_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOD_REQ_DATA_Ttag
{
  uint32_t              ulStructVersion;
  PNM_AP_DEVICEHANDLE_T usDeviceHandle;
};

/** request packet */
typedef struct PNM_AP_CFG_GET_IOD_REQ_Ttag PNM_AP_CFG_GET_IOD_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOD_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T           tHead;
  PNM_AP_CFG_GET_IOD_REQ_DATA_T tData;
};

/** confirmation packet */
typedef struct PNM_AP_CFG_GET_IOD_CNF_Ttag PNM_AP_CFG_GET_IOD_CNF_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOD_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T       tHead;
  PNM_AP_CFG_IOD_REQ_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_GET_IOD_REQ_T tReq;
  /** confirmation packet */
  PNM_AP_CFG_GET_IOD_CNF_T tCnf;
} PNM_AP_CFG_GET_IOD_PCK_T;

/*******************************************************************************
 GET AR PRM PARAMETERS
*******************************************************************************/

#define PNM_AP_CFG_GET_AR_PRM_STRUCT_VERSION_1 (0x0001)

/*
PNM_AP_CMD_CFG_GET_AR_PRM_LIST_REQ
    return all "ParameterTypes" that have been configured for IO Device refwered to by "DeviceHandle"
*/

typedef struct PNM_AP_CFG_GET_AR_PRM_LIST_REQ_DATA_Ttag PNM_AP_CFG_GET_AR_PRM_LIST_REQ_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_AR_PRM_LIST_REQ_DATA_Ttag
{
  uint32_t ulStructVersion;
  uint16_t uiDeviceHandle;
};

/** request packet */
typedef struct PNM_AP_CFG_GET_AR_PRM_LIST_REQ_Ttag PNM_AP_CFG_GET_AR_PRM_LIST_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_AR_PRM_LIST_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T                   tHead;
  PNM_AP_CFG_GET_AR_PRM_LIST_REQ_DATA_T tData;
};

#define PNM_AP_MAX_AR_PRM 4

typedef struct PNM_AP_CFG_GET_AR_PRM_LIST_CNF_DATA_Ttag PNM_AP_CFG_GET_AR_PRM_LIST_CNF_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_AR_PRM_LIST_CNF_DATA_Ttag
{
    uint32_t ulStructVersion;
    uint16_t uiPrmTypes[PNM_AP_MAX_AR_PRM];
    uint16_t uiPrmTypesSize;
};

/** confirmation packet */
typedef struct PNM_AP_CFG_GET_AR_PRM_LIST_CNF_Ttag PNM_AP_CFG_GET_AR_PRM_LIST_CNF_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_AR_PRM_LIST_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T                   tHead;
  PNM_AP_CFG_GET_AR_PRM_LIST_CNF_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_GET_AR_PRM_LIST_REQ_T tReq;
  /** confirmation packet */
  PNM_AP_CFG_GET_AR_PRM_LIST_CNF_T tCnf;
} PNM_AP_CFG_GET_AR_PRM_LIST_PCK_T;

/*
PNM_AP_CMD_CFG_GET_AR_PRM_REQ
    return the requested configuration for "ParameterType" given in request for AR referenced by using its "DeviceHandle"
*/

typedef struct PNM_AP_CFG_GET_AR_PRM_REQ_DATA_Ttag PNM_AP_CFG_GET_AR_PRM_REQ_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_AR_PRM_REQ_DATA_Ttag {
  uint32_t ulStructVersion;
  uint16_t uiDeviceHandle;
  uint16_t uiPrmType;

  /* PNM_AP_CFG_AR_PRM_IDENT_ARVENDORBLOCKREQ */
  uint16_t uiVendorHandle;

  /* PNM_AP_CFG_AR_PRM_IDENT_IRDATA */
  /* Nothing */
};

/** request packet */
typedef struct PNM_AP_CFG_GET_AR_PRM_REQ_Ttag PNM_AP_CFG_GET_AR_PRM_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_AR_PRM_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T              tHead;
  PNM_AP_CFG_GET_AR_PRM_REQ_DATA_T tData;
};

/** confirmation packet */
typedef struct PNM_AP_CFG_GET_AR_PRM_CNF_Ttag PNM_AP_CFG_GET_AR_PRM_CNF_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_AR_PRM_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T      tHead;
  PNM_AP_CFG_AR_PRM_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_GET_AR_PRM_REQ_T tReq;
  /** confirmation packet */
  PNM_AP_CFG_GET_AR_PRM_CNF_T tCnf;
} PNM_AP_CFG_GET_AR_PRM_PCK_T;

/*******************************************************************************
 GET IOCR PARAMETERS
*******************************************************************************/

/*
PNM_AP_CMD_CFG_GET_IOCR_REQ
    return IOCR configuration of the requested IOCR
*/

typedef struct PNM_AP_CFG_GET_IOCR_REQ_DATA_Ttag PNM_AP_CFG_GET_IOCR_REQ_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOCR_REQ_DATA_Ttag
{
  /** structure version of this structure */
  uint32_t                ulStructVersion;

  PNM_AP_IOCR_HANDLE_T    usIocrHandle;
};

/** request packet */
typedef struct PNM_AP_CFG_GET_IOCR_REQ_Ttag PNM_AP_CFG_GET_IOCR_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOCR_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T            tHead;
  PNM_AP_CFG_GET_IOCR_REQ_DATA_T tData;
};

/** confirmation packet */
typedef struct PNM_AP_CFG_GET_IOCR_CNF_Ttag   PNM_AP_CFG_GET_IOCR_CNF_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_IOCR_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T    tHead;
  PNM_AP_CFG_IOCR_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_GET_IOCR_REQ_T tReq;
  /** confirmation packet */
  PNM_AP_CFG_GET_IOCR_CNF_T tCnf;
} PNM_AP_CFG_GET_IOCR_PCK_T;

/*******************************************************************************
 GET SUBMODULES PARAMETERS
*******************************************************************************/

/*
PNM_AP_CMD_CFG_GET_SUBMODULE_REQ
    return submodule parameters of the requested submodule
*/

typedef struct PNM_AP_CFG_GET_SUBMODULE_REQ_DATA_Ttag PNM_AP_CFG_GET_SUBMODULE_REQ_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_SUBMODULE_REQ_DATA_Ttag
{
  /** requested structure version for confirmation */
  uint32_t                  ulStructVersion;
  /** handle of submodule to retrieve */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
};

/** request packet */
typedef struct PNM_AP_CFG_GET_SUBMODULE_REQ_Ttag PNM_AP_CFG_GET_SUBMODULE_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_SUBMODULE_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T                 tHead;
  PNM_AP_CFG_GET_SUBMODULE_REQ_DATA_T tData;
};

/** confirmation packet */
typedef struct PNM_AP_CFG_GET_SUBMODULE_CNF_Ttag PNM_AP_CFG_GET_SUBMODULE_CNF_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_SUBMODULE_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T         tHead;
  PNM_AP_CFG_SUBMODULE_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_GET_SUBMODULE_REQ_T tReq;
  /** confirmation packet */
  PNM_AP_CFG_GET_SUBMODULE_CNF_T tCnf;
} PNM_AP_CFG_GET_SUBMODULE_PCK_T;

/*******************************************************************************
 GET RECORD PARAMETERS
*******************************************************************************/

/*
   shall return ONLY the following parameters
 * PNM_AP_CFG_PRM_ISOCHRONOUSCONTROLLERDATA
 * PNM_AP_CFG_PRM_ISOCHRONOUSMODEDATA
 * PNM_AP_CFG_PRM_RECORD (to deliver this Type in Requestor has to provide the index)

   other parameter types can not be read by application

PNM_AP_CMD_CFG_GET_RECORD_DATA_REQ
   get records and parameters for a specific submodule
*/

typedef struct PNM_AP_CFG_GET_RECORD_REQ_DATA_Ttag PNM_AP_CFG_GET_RECORD_REQ_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_RECORD_REQ_DATA_Ttag
{
  /** requested structure version for confirmation */
  uint32_t                  ulStructVersion;
  /** handle of submodule to retrieve */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;

  uint16_t uiPrmType;
};

/** request packet */
typedef struct PNM_AP_CFG_GET_RECORD_REQ_Ttag PNM_AP_CFG_GET_RECORD_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_RECORD_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T tHead;
  PNM_AP_CFG_GET_RECORD_REQ_DATA_T tData;
};

/** confirmation packet */
typedef struct PNM_AP_CFG_GET_RECORD_CNF_Ttag PNM_AP_CFG_GET_RECORD_CNF_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_GET_RECORD_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T tHead;
  PNM_AP_CFG_RECORD_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_CFG_GET_RECORD_REQ_T tReq;
  /** confirmation packet */
  PNM_AP_CFG_GET_RECORD_CNF_T tCnf;
} PNM_AP_CFG_GET_RECORD_PCK_T;

/*******************************************************************************
 Configure OEM Parameter Service
*******************************************************************************/

typedef struct PNM_AP_CFG_OEMPRM_VERSION_Ttag PNM_AP_CFG_OEMPRM_VERSION_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_OEMPRM_VERSION_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;
  /** the hardware version number */
  uint16_t usHWVersion;
  /** the software version prefix */
  uint8_t  bPrefix;
  /** padding */
  uint8_t  bPadding;
  /** the major version number */
  uint16_t usVersionMajor;
  /** the minor version number */
  uint16_t usVersionMinor;
  /** the revision version number */
  uint16_t usVersionBugfix;
};

typedef struct PNM_AP_CFG_OEMPRM_SERIAL_NUMBER_Ttag PNM_AP_CFG_OEMPRM_SERIAL_NUMBER_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_OEMPRM_SERIAL_NUMBER_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;
  /** serial number */
  uint8_t  abSerialNumber[16];
};

typedef struct PNM_AP_CFG_OEMPRM_ORDER_ID_Ttag PNM_AP_CFG_OEMPRM_ORDER_ID_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_OEMPRM_ORDER_ID_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;
  /** Order ID */
  uint8_t  abOrderID[20];
};

typedef struct PNM_AP_CFG_OEMPRM_ALARM_HANDLING_Ttag PNM_AP_CFG_OEMPRM_ALARM_HANDLING_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_OEMPRM_ALARM_HANDLING_Ttag
{
  /** structure version of this structure */
  uint32_t ulStructVersion;
  /** Automatic Alarm handling flags
   *  The definition of the ulAlarmHandlingFlags are \ref PNM_AP_CFG_ALARM_HANDLING_FLAGS_Etag enumeration. */
  uint32_t ulAlarmHandlingFlags;
};

typedef struct PNM_AP_CFG_OEMPRM_DEVICE_IDENTITY_Ttag PNM_AP_CFG_OEMPRM_DEVICE_IDENTITY_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_OEMPRM_DEVICE_IDENTITY_Ttag
{
  /** Version of this structure */
  uint32_t ulStructVersion;
  /** Vendor Id */
  uint16_t usVendorId;
  /** Device Id */
  uint16_t usDeviceId;
  /** Device Instance */
  uint16_t usInstance;
};

/** Enumeration of known oem parameter types */
enum PNM_AP_CFG_OEMPRM_TYPE_Etag
{
  PNM_AP_CFG_OEMPRM_VERSION                 =  1,
  PNM_AP_CFG_OEMPRM_SERIAL_NUMBER           =  2,
  PNM_AP_CFG_OEMPRM_ORDER_ID                =  3,
  PNM_AP_CFG_OEMPRM_ALARM_HANDLING          =  4,
  /** Available since V3.2.0.0 */
  PNM_AP_CFG_OEMPRM_DEVICE_IDENTITY         =  5,
};

typedef enum PNM_AP_CFG_OEMPRM_TYPE_Etag PNM_AP_CFG_OEMPRM_TYPE_E;
/* parameter union */
typedef union PNM_AP_CFG_OEMPRM_UNION_Ttag PNM_AP_CFG_OEMPRM_UNION_T;
union PNM_AP_CFG_OEMPRM_UNION_Ttag
{
  PNM_AP_CFG_OEMPRM_VERSION_T             tVersion;
  PNM_AP_CFG_OEMPRM_SERIAL_NUMBER_T       tSerialNumber;
  PNM_AP_CFG_OEMPRM_ORDER_ID_T            tOrderId;
  PNM_AP_CFG_OEMPRM_ALARM_HANDLING_T      tAlarmHandling;
  PNM_AP_CFG_OEMPRM_DEVICE_IDENTITY_T     tDeviceIdentity;
};

/* allowed values for field ulStructVersion */
#define PNM_AP_CFG_OEMPRM_STRUCT_VERSION_1    (0x0001)

/** request packet data */
typedef struct PNM_AP_CFG_OEMPRM_DATA_Ttag PNM_AP_CFG_OEMPRM_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_OEMPRM_DATA_Ttag
{
  /** structure version of this structure */
  uint32_t                   ulStructVersion;

  /** identifier see \ref PNM_AP_CFG_PRM_TYPE_E */
  uint16_t                   usPrmType;
  /** Reserved (Padding) */
  uint16_t                   usPadding;
  /** parameter data */
  PNM_AP_CFG_OEMPRM_UNION_T  uData;
} ;

/** request packet */
typedef struct PNM_AP_CFG_OEMPRM_REQ_Ttag PNM_AP_CFG_OEMPRM_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_CFG_OEMPRM_REQ_Ttag
{
  /** packet header */
  PNM_AP_PCK_HEADER_T                    tHead;
  /** packet data */
  PNM_AP_CFG_OEMPRM_DATA_T               tData;
};

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T            PNM_AP_CFG_OEMPRM_CNF_T;

/** packet union */
typedef union PNM_AP_CFG_OEMPRM_PCK_Ttag PNM_AP_CFG_OEMPRM_PCK_T;

union PNM_AP_CFG_OEMPRM_PCK_Ttag
{
  /** request packet */
  PNM_AP_CFG_OEMPRM_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_CFG_OEMPRM_CNF_T    tCnf;
};

/******************************************************************************

88888888ba                                     88
88      "8b                             ,d     ""
88      ,8P                             88
88aaaaaa8P'  88       88  8b,dPPYba,  MM88MMM  88  88,dPYba,,adPYba,    ,adPPYba,
88""""88'    88       88  88P'   `"8a   88     88  88P'   "88"    "8a  a8P_____88
88    `8b    88       88  88       88   88     88  88      88      88  8PP"""""""
88     `8b   "8a,   ,a88  88       88   88,    88  88      88      88  "8b,   ,aa
88      `8b   `"YbbdP'Y8  88       88   "Y888  88  88      88      88   `"Ybbd8"'

patorjk.com/software/taag --> Font Univers
******************************************************************************/

/*******************************************************************************
 Establish DAAR Service
*******************************************************************************/
/* allowed values for field ulStructVersion */
#define PNM_AP_ESTABLISH_DAAR_STRUCT_VERSION_1    (0x0001)

/* request packet data */
typedef struct PNM_AP_ESTABLISH_DAAR_DATA_Ttag PNM_AP_ESTABLISH_DAAR_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_ESTABLISH_DAAR_DATA_Ttag
{
  /** structure version of this structure */
  uint32_t              ulStructVersion;
  /** device handle of DA-AR. (Always 0xF000) */
  PNM_AP_DEVICEHANDLE_T usDeviceHandle;
  /** AR Type see \ref PNM_AP_CFG_IOD_AR_TYPE_E
   * Always PNM_AP_CFG_IOD_AR_TYPE_SUPERVISOR for DA-AR*/
  uint8_t               bARType;
  /** padding, set to 0 */
  uint8_t               bPadding;
  /** flags see \ref PNM_AP_CFG_IOD_FLAG_E */
  uint32_t              ulFlags;
  /** AR UUID - has to be unique over all configured IO-Devices
      \par See Spec:
       Coding of the field ARUUID
  */
  PNM_UUID_T            tArUuid;
  /** AR Properties see \ref PNM_AP_CFG_IOD_AR_PROP_FLAGS_E
      \par See Spec:
       Coding of the field ARProperties
  */
  uint32_t              ulArProperties;
  /** VendorID */
  uint16_t              usVendorID;
  /** DeviceID */
  uint16_t              usDeviceID;
  /** InstanceID (GSDML parameter ObjectUUID Instance) */
  uint16_t              usInstanceID;
  /** reserved */
  uint16_t              ausReserved[3];
  /** NameOfStation of the IO Device
      zero padded: length determined by searching first zero
      \note  The GSDML Element \c OrderNumber.
      \par See Spec:
       Coding of the field NameOfStationValue
  */
  uint8_t               abNameOfStation[240];
  /** IP address */
  uint32_t              ulIPAddr;
  /** network mask */
  uint32_t              ulNetworkMask;
  /** Gateway address */
  uint32_t              ulGatewayAddr;
};

/* request packet */
typedef struct PNM_AP_ESTABLISH_DAAR_REQ_Ttag PNM_AP_ESTABLISH_DAAR_REQ_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_ESTABLISH_DAAR_REQ_Ttag
{
  /** packet header */
  PNM_AP_PCK_HEADER_T               tHead;
  /** packet data */
  PNM_AP_ESTABLISH_DAAR_DATA_T      tData;
} ;

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T          PNM_AP_ESTABLISH_DAAR_CNF_T;

/** packet union */
typedef union PNM_AP_ESTABLISH_DAAR_PCK_Ttag PNM_AP_ESTABLISH_DAAR_PCK_T;
union PNM_AP_ESTABLISH_DAAR_PCK_Ttag
{
  /** request packet */
  PNM_AP_ESTABLISH_DAAR_REQ_T  tReq;
  /** confirmation packet */
  PNM_AP_ESTABLISH_DAAR_CNF_T  tCnf;
};

/*******************************************************************************
 Release DAAR Service
*******************************************************************************/


/** usable flags in ulFlags */
enum PNM_AP_RELEASE_DAAR_FLAGS_Etag
{
  PNM_AP_RELEASE_DAAR_RELEASE = PNM_AP_CFG_IOD_FLAG_DISABLED,
};

typedef enum PNM_AP_RELEASE_DAAR_FLAGS_Etag PNM_AP_RELEASE_DAAR_FLAGS_E;

typedef struct PNM_AP_RELEASE_DAAR_REQ_DATA_Ttag PNM_AP_RELEASE_DAAR_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_RELEASE_DAAR_REQ_DATA_Ttag
{
  /** unique handle of this IO-Device defined by the sender of this packet */
  PNM_AP_DEVICEHANDLE_T usDeviceHandle;
  /* padding for alignment */
  uint16_t              usPadding;
  /** status flags see \ref PNM_AP_SET_AR_STATUS_E */
  uint32_t              ulFlags;
};

typedef struct PNM_AP_RELEASE_DAAR_REQ_Ttag PNM_AP_RELEASE_DAAR_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_RELEASE_DAAR_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T            tHead;

  PNM_AP_RELEASE_DAAR_REQ_DATA_T tData;
};

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T PNM_AP_RELEASE_DAAR_CNF_T;

/** packet union */
typedef union PNM_AP_RELEASE_DAAR_PCK_Ttag PNM_AP_RELEASE_DAAR_PCK_T;
union PNM_AP_RELEASE_DAAR_PCK_Ttag
{
  /** request packet */
  PNM_AP_RELEASE_DAAR_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_RELEASE_DAAR_CNF_T    tCnf;
};


/*******************************************************************************
 GET AR VENDORBLOCK RESPONSE SERVICE
*******************************************************************************/

/** request packet */
typedef struct PNM_AP_GET_ARVENDORBLOCK_RESPONSE_REQ_DATA_Ttag PNM_AP_GET_ARVENDORBLOCK_RESPONSE_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_GET_ARVENDORBLOCK_RESPONSE_REQ_DATA_Ttag
{
  /** handle of the AR VendorBlock, was given at runtime */
  PNM_AP_ARVENDORBLOCKHANDLE_T usArVendorBlockHandle;
};

typedef struct PNM_AP_GET_ARVENDORBLOCK_RESPONSE_REQ_Ttag PNM_AP_GET_ARVENDORBLOCK_RESPONSE_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_GET_ARVENDORBLOCK_RESPONSE_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T                          tHead;
  PNM_AP_GET_ARVENDORBLOCK_RESPONSE_REQ_DATA_T tData;
};

/** confirmation packet */
typedef struct PNM_AP_GET_ARVENDORBLOCK_RESPONSE_CNF_DATA_Ttag PNM_AP_GET_ARVENDORBLOCK_RESPONSE_CNF_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_GET_ARVENDORBLOCK_RESPONSE_CNF_DATA_Ttag
{
  /** handle of this AR VendorBlock, required to read out the Response at runtime */
  PNM_AP_ARVENDORBLOCKHANDLE_T  usArVendorBlockHandle;
  /** AP Structure Identifier according to GSDML */
  uint16_t                      usAPStructureIdentifier;
  /** API according to GSDML */
  uint32_t                      ulApi;
  /** Array of the VendorBlock data, sizeof array depend on parameter usDataLen  */
  uint8_t                       abData[ARRAYS_OF_LENGTH_ZERO];
};

typedef struct PNM_AP_GET_ARVENDORBLOCK_RESPONSE_CNF_Ttag PNM_AP_GET_ARVENDORBLOCK_RESPONSE_CNF_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_GET_ARVENDORBLOCK_RESPONSE_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T                          tHead;
  PNM_AP_GET_ARVENDORBLOCK_RESPONSE_CNF_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_GET_ARVENDORBLOCK_RESPONSE_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_GET_ARVENDORBLOCK_RESPONSE_CNF_T    tCnf;
} PNM_AP_GET_ARVENDORBLOCK_RESPONSE_PCK_T;



/*******************************************************************************
 READ RECORD SERVICE
*******************************************************************************/

/** request packet */
typedef struct PNM_AP_READ_RECORD_SUBM_REQ_DATA_Ttag PNM_AP_READ_RECORD_SUBM_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_READ_RECORD_SUBM_REQ_DATA_Ttag
{
  /** submodule handle to read from */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
  /** index to read */
  uint16_t                  usIndex;
  /** max expected data length to request */
  uint32_t                  ulMaxReadLen;
};

typedef struct PNM_AP_READ_RECORD_SUBM_REQ_Ttag PNM_AP_READ_RECORD_SUBM_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_READ_RECORD_SUBM_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T                tHead;
  PNM_AP_READ_RECORD_SUBM_REQ_DATA_T tData;
};

/** confirmation packet */
typedef struct PNM_AP_READ_RECORD_SUBM_CNF_DATA_Ttag PNM_AP_READ_RECORD_SUBM_CNF_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_READ_RECORD_SUBM_CNF_DATA_Ttag
{
  /** submodule handle to read from */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
  /** index to read */
  uint16_t                  usIndex;
  /** data length returned */
  uint32_t                  ulDataLen;
  /** Profinet error code */
  uint32_t                  ulPnio;
  /** Additional Value 1 */
  uint16_t                  usAddVal1;
  /** Additional Value 2 */
  uint16_t                  usAddVal2;
  /** record data, amount of bytes see usDataLen */
  uint8_t                   abRecordData[];
};

typedef struct PNM_AP_READ_RECORD_SUBM_CNF_Ttag PNM_AP_READ_RECORD_SUBM_CNF_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_READ_RECORD_SUBM_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T                tHead;
  PNM_AP_READ_RECORD_SUBM_CNF_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_READ_RECORD_SUBM_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_READ_RECORD_SUBM_CNF_T    tCnf;
} PNM_AP_READ_RECORD_SUBM_PCK_T;


/*******************************************************************************
 WRITE RECORD SERVICE
*******************************************************************************/

/** request packet */
typedef struct PNM_AP_WRITE_RECORD_SUBM_REQ_DATA_Ttag PNM_AP_WRITE_RECORD_SUBM_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_WRITE_RECORD_SUBM_REQ_DATA_Ttag
{
  /** submodule handle to read from */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
  /** index to read */
  uint16_t                  usIndex;
  /** data length to write */
  uint32_t                  ulDataLen;
  /** record data, amount of bytes see usDataLen */
  uint8_t                   abRecordData[ARRAYS_OF_LENGTH_ZERO];

};

typedef struct PNM_AP_WRITE_RECORD_SUBM_REQ_Ttag PNM_AP_WRITE_RECORD_SUBM_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_WRITE_RECORD_SUBM_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T                 tHead;
  PNM_AP_WRITE_RECORD_SUBM_REQ_DATA_T tData;
};

/** confirmation packet */
typedef struct PNM_AP_WRITE_RECORD_SUBM_CNF_DATA_Ttag PNM_AP_WRITE_RECORD_SUBM_CNF_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_WRITE_RECORD_SUBM_CNF_DATA_Ttag
{
  /** submodule handle to read from */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
  /** index to read */
  uint16_t                  usIndex;
  /** Profinet error code */
  uint32_t                  ulPnio;
  /** Additional Value 1 */
  uint16_t                  usAddVal1;
  /** Additional Value 2 */
  uint16_t                  usAddVal2;
};

typedef struct PNM_AP_WRITE_RECORD_SUBM_CNF_Ttag PNM_AP_WRITE_RECORD_SUBM_CNF_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_WRITE_RECORD_SUBM_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T                 tHead;
  PNM_AP_WRITE_RECORD_SUBM_CNF_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_WRITE_RECORD_SUBM_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_WRITE_RECORD_SUBM_CNF_T    tCnf;
} PNM_AP_WRITE_RECORD_SUBM_PCK_T;


/*******************************************************************************
 READ IMPLICIT RECORD SERVICE
*******************************************************************************/

/** request packet */
typedef struct PNM_AP_READ_IMPLICIT_RECORD_REQ_DATA_Ttag PNM_AP_READ_IMPLICIT_RECORD_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_READ_IMPLICIT_RECORD_REQ_DATA_Ttag
{
  /** IP-address of the IO-Device to read from */
  uint32_t      ulIPAddr;
  /** VendorID */
  uint16_t      usVendorId;
  /** DeviceID */
  uint16_t      usDeviceId;
  /** ObjectUUID InstanceID */
  uint16_t      usInstanceId;
  /** reserved */
  uint16_t      usReserved1;
  /** API to read from */
  uint32_t      ulApi;
  /** slot to read from*/
  uint16_t      usSlot;
  /** subslot to read from */
  uint16_t      usSubslot;
  /** index to read */
  uint16_t      usIndex;
  /** reserved */
  uint16_t      usReserved2;
  /** max expected data length to request */
  uint32_t      ulMaxLenToRead;
};

typedef struct PNM_AP_READ_IMPLICIT_RECORD_REQ_Ttag PNM_AP_READ_IMPLICIT_RECORD_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_READ_IMPLICIT_RECORD_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T                    tHead;
  PNM_AP_READ_IMPLICIT_RECORD_REQ_DATA_T tData;
};

/** confirmation packet */
typedef struct PNM_AP_READ_IMPLICIT_RECORD_CNF_DATA_Ttag PNM_AP_READ_IMPLICIT_RECORD_CNF_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_READ_IMPLICIT_RECORD_CNF_DATA_Ttag
{
  /** IP-address of the IO-Device to read from */
  uint32_t      ulIPAddr;
  /** VendorID */
  uint16_t      usVendorId;
  /** DeviceID */
  uint16_t      usDeviceId;
  /** ObjectUUID InstanceID */
  uint16_t      usInstanceId;
  /** reserved */
  uint16_t      usReserved1;
  /** API to read from */
  uint32_t      ulApi;
  /** slot to read from*/
  uint16_t      usSlot;
  /** subslot to read from */
  uint16_t      usSubslot;
  /** index to read */
  uint16_t      usIndex;
  /** reserved */
  uint16_t      usReserved2;
  /** max expected data length to request */
  uint32_t      ulLenRead;
  /** Profinet error code */
  uint32_t      ulPnio;
  /** Additional Value 1 */
  uint16_t      usAddVal1;
  /** Additional Value 2 */
  uint16_t      usAddVal2;
  /** record data, amount of bytes see usDataLen */
  uint8_t       abRecordData[ARRAYS_OF_LENGTH_ZERO];
};

typedef struct PNM_AP_READ_IMPLICIT_RECORD_CNF_Ttag PNM_AP_READ_IMPLICIT_RECORD_CNF_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_READ_IMPLICIT_RECORD_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T                    tHead;
  PNM_AP_READ_IMPLICIT_RECORD_CNF_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_READ_IMPLICIT_RECORD_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_READ_IMPLICIT_RECORD_CNF_T    tCnf;
} PNM_AP_READ_IMPLICIT_RECORD_PCK_T;


/*******************************************************************************
 ALARM INDICATION SERVICE
*******************************************************************************/

/** indication packet */
typedef struct PNM_AP_RECV_ALARM_IND_DATA_Ttag PNM_AP_RECV_ALARM_IND_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_RECV_ALARM_IND_DATA_Ttag
{
  /** IO-Device handle */
  PNM_AP_DEVICEHANDLE_T     usDeviceHandle;
  /** submodule handle that issued the alarm */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
  /** alarm type */
  uint16_t                  usAlarmType;
  /** alarm priority */
  uint16_t                  usAlarmPriority;
  /** API of submodule */
  uint32_t                  ulApi;
  /** slot of submodule */
  uint16_t                  usSlot;
  /** subslot of submodule */
  uint16_t                  usSubslot;
  /** alarm specified */
  uint16_t                  usAlarmSpecifier;
  /** alarm data length */
  /* if value is 0 the field abAlarmData do not contain valid data */
  uint16_t                  usAlarmDataLen;
  /** alarm data, amount of bytes see usAlarmDataLen */
  /* the User Structure Identifier (if any) is represented by the first 2 bytes in the array */
  uint8_t                   abAlarmData[PNM_AP_MAX_ALARM_DATA_LENGTH];
};

typedef struct PNM_AP_RECV_ALARM_IND_Ttag PNM_AP_RECV_ALARM_IND_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_RECV_ALARM_IND_Ttag
{
  PNM_AP_PCK_HEADER_T          tHead;
  PNM_AP_RECV_ALARM_IND_DATA_T tData;
};

/** response packet */
typedef struct PNM_AP_RECV_ALARM_RES_DATA_Ttag PNM_AP_RECV_ALARM_RES_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_RECV_ALARM_RES_DATA_Ttag
{
  /** IO-Device handle */
  PNM_AP_DEVICEHANDLE_T     usDeviceHandle;
  /** submodule handle that issued the alarm */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
  /** alarm type */
  uint16_t                  usAlarmType;
  /** alarm priority, copied from indication */
  uint16_t                  usAlarmPriority;
};

typedef struct PNM_AP_RECV_ALARM_RES_Ttag PNM_AP_RECV_ALARM_RES_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_RECV_ALARM_RES_Ttag
{
  PNM_AP_PCK_HEADER_T               tHead;
  PNM_AP_RECV_ALARM_RES_DATA_T      tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_RECV_ALARM_IND_T    tInd;
  /** confirmation packet */
  PNM_AP_RECV_ALARM_RES_T    tRes;
} PNM_AP_RECV_ALARM_PCK_T;



/*******************************************************************************
 DIAGNOSIS INDICATION SERVICE
*******************************************************************************/

typedef struct PNM_AP_CHANNEL_DIAGNOSIS_DATA_Ttag PNM_AP_CHANNEL_DIAGNOSIS_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_CHANNEL_DIAGNOSIS_DATA_Ttag
{
  uint16_t usChannelNumber;
  uint16_t usChannelProperties;
  uint16_t usChannelErrorType;
};

typedef struct PNM_AP_EXT_CHANNEL_DIAGNOSIS_DATA_Ttag PNM_AP_EXT_CHANNEL_DIAGNOSIS_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_EXT_CHANNEL_DIAGNOSIS_DATA_Ttag
{
  uint16_t usChannelNumber;
  uint16_t usChannelProperties;
  uint16_t usChannelErrorType;
  uint16_t usExtChannelErrorType;
  uint32_t ulExtChannelAddValue;
};

typedef struct PNM_AP_QUALIFIED_CHANNEL_DIAGNOSIS_DATA_Ttag PNM_AP_QUALIFIED_CHANNEL_DIAGNOSIS_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_QUALIFIED_CHANNEL_DIAGNOSIS_DATA_Ttag
{
  uint16_t usChannelNumber;
  uint16_t usChannelProperties;
  uint16_t usChannelErrorType;
  uint16_t usExtChannelErrorType;
  uint32_t ulExtChannelAddValue;
  uint32_t ulQualifiedChannelQualifier;
};

typedef struct PNM_AP_MANUF_SPECIFIC_DIAGNOSIS_DATA_Ttag PNM_AP_MANUF_SPECIFIC_DIAGNOSIS_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_MANUF_SPECIFIC_DIAGNOSIS_DATA_Ttag
{
  uint8_t abData[PNM_AP_MAX_ALARM_DATA_LENGTH];
};

#define PNM_AP_MAX_CHANNELDIAG_CNT          (PNM_AP_MAX_ALARM_DATA_LENGTH / sizeof(PNM_AP_CHANNEL_DIAGNOSIS_DATA_T))
#define PNM_AP_MAX_EXTCHANNELDIAG_CNT       (PNM_AP_MAX_ALARM_DATA_LENGTH/ sizeof(PNM_AP_EXT_CHANNEL_DIAGNOSIS_DATA_T))
#define PNM_AP_MAX_QUALIFIEDCHANNELDIAG_CNT (PNM_AP_MAX_ALARM_DATA_LENGTH / sizeof(PNM_AP_QUALIFIED_CHANNEL_DIAGNOSIS_DATA_T))

typedef union PNM_AP_DIAGNOSIS_DATA_Utag PNM_AP_DIAGNOSIS_DATA_U;

__PACKED_PRE union __PACKED_POST PNM_AP_DIAGNOSIS_DATA_Utag
{
  PNM_AP_CHANNEL_DIAGNOSIS_DATA_T           tChannelDiag;
  PNM_AP_CHANNEL_DIAGNOSIS_DATA_T           atChannelDiag[PNM_AP_MAX_CHANNELDIAG_CNT];

  PNM_AP_EXT_CHANNEL_DIAGNOSIS_DATA_T       tExtChannelDiag;
  PNM_AP_EXT_CHANNEL_DIAGNOSIS_DATA_T       atExtChannelDiag[PNM_AP_MAX_EXTCHANNELDIAG_CNT];

  PNM_AP_QUALIFIED_CHANNEL_DIAGNOSIS_DATA_T tQualifChannelDiag;
  PNM_AP_QUALIFIED_CHANNEL_DIAGNOSIS_DATA_T atQualifChannelDiag[PNM_AP_MAX_QUALIFIEDCHANNELDIAG_CNT];

  PNM_AP_MANUF_SPECIFIC_DIAGNOSIS_DATA_T    tManufSpecificDiag;
};

/** indication packet */
typedef struct PNM_AP_RECV_DIAGNOSIS_IND_DATA_Ttag PNM_AP_RECV_DIAGNOSIS_IND_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_RECV_DIAGNOSIS_IND_DATA_Ttag
{
  /** IO-Device handle */
  PNM_AP_DEVICEHANDLE_T     usDeviceHandle;
  /** submodule handle that issued the alarm */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
  /** alarm type */
  uint16_t                  usAlarmType;
  /** alarm priority */
  uint16_t                  usAlarmPriority;
  /** API of submodule */
  uint32_t                  ulApi;
  /** slot of submodule */
  uint16_t                  usSlot;
  /** subslot of submodule */
  uint16_t                  usSubslot;
  /** alarm specified */
  uint16_t                  usAlarmSpecifier;
  /** alarm data length */
  /* if value is 0 the field abAlarmData do not contain valid data */
  uint16_t                  usAlarmDataLen;
  /** UserStructureIdentifier */
  uint16_t                  usUSI;
  /** union representing the Diagnosis alarm data */
  /* USI specified the content of uData */
  /* 0x8000     -> tChannelDiag */
  /* 0x8002     -> tExtChannelDiag */
  /* 0x8003     -> tQualifChannelDiag */
  /* 0...0x7FFF -> tManufSpecificDiag */
  PNM_AP_DIAGNOSIS_DATA_U   uData;
};

typedef struct PNM_AP_RECV_DIAGNOSIS_IND_Ttag PNM_AP_RECV_DIAGNOSIS_IND_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_RECV_DIAGNOSIS_IND_Ttag
{
  PNM_AP_PCK_HEADER_T              tHead;
  PNM_AP_RECV_DIAGNOSIS_IND_DATA_T tData;
};

/** response packet */
typedef struct PNM_AP_RECV_DIAGNOSIS_RES_DATA_Ttag PNM_AP_RECV_DIAGNOSIS_RES_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_RECV_DIAGNOSIS_RES_DATA_Ttag
{
  /** IO-Device handle */
  PNM_AP_DEVICEHANDLE_T     usDeviceHandle;
  /** submodule handle that issued the alarm */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
  /** alarm type */
  uint16_t                  usAlarmType;
  /** alarm priority, copied from indication */
  uint16_t                  usAlarmPriority;
};

typedef struct PNM_AP_RECV_DIAGNOSIS_RES_Ttag PNM_AP_RECV_DIAGNOSIS_RES_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_RECV_DIAGNOSIS_RES_Ttag
{
  PNM_AP_PCK_HEADER_T               tHead;
  PNM_AP_RECV_DIAGNOSIS_RES_DATA_T      tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_RECV_DIAGNOSIS_IND_T    tInd;
  /** confirmation packet */
  PNM_AP_RECV_DIAGNOSIS_RES_T    tRes;
} PNM_AP_RECV_DIAGNOSIS_PCK_T;

/*******************************************************************************
 ALARM / Diagnosis Acknowledge Service
*******************************************************************************/

/** indication packet */
typedef struct PNM_AP_ACK_ALARM_REQ_DATA_Ttag PNM_AP_ACK_ALARM_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_ACK_ALARM_REQ_DATA_Ttag
{
  /** IO-Device handle */
  PNM_AP_DEVICEHANDLE_T     usDeviceHandle;
  /** submodule handle that issued the alarm */
  PNM_AP_SUBMODULE_HANDLE_T usSubmoduleHandle;
  /** alarm type */
  uint16_t                  usAlarmType;
  /** alarm priority */
  uint16_t                  usAlarmPriority;
  /** status */
  uint32_t                  ulPnio;
};

typedef struct PNM_AP_ACK_ALARM_REQ_Ttag PNM_AP_ACK_ALARM_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_ACK_ALARM_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T         tHead;
  PNM_AP_ACK_ALARM_REQ_DATA_T tData;
};

/** response packet */
typedef PNM_AP_EMTPY_PCK_T PNM_AP_ACK_ALARM_CNF_T;

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_ACK_ALARM_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_ACK_ALARM_CNF_T    tCnf;
} PNM_AP_ACK_ALARM_PCK_T;

/*******************************************************************************
 DCP GENERIC DEFINITIONS
*******************************************************************************/

#define PNIO_APCTL_DCP_QUALIFIER_STORE_TEMPORARY                0x0
#define PNIO_APCTL_DCP_QUALIFIER_STORE_PERMANENT                0x1

#define PNIO_APCTL_DCP_QUALIFIER_RESETTOFACTORY_APPLICATION    (1 << 1)
#define PNIO_APCTL_DCP_QUALIFIER_RESETTOFACTORY_COMMUNICATION  (2 << 1)
#define PNIO_APCTL_DCP_QUALIFIER_RESETTOFACTORY_ENGINEERING    (3 << 1)
#define PNIO_APCTL_DCP_QUALIFIER_RESETTOFACTORY_ALL            (4 << 1)
#define PNIO_APCTL_DCP_QUALIFIER_RESETTOFACTORY_DEVICE         (8 << 1)
#define PNIO_APCTL_DCP_QUALIFIER_RESETTOFACTORY_RESTORE        (9 << 1)

/** confirmation packet */
typedef struct PNM_AP_DCP_SET_CNF_DATA_Ttag PNM_AP_DCP_SET_CNF_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_CNF_DATA_Ttag
{
  /** DCP status */
  uint8_t bDcpError;
};

typedef struct PNM_AP_DCP_SET_CNF_Ttag PNM_AP_DCP_SET_CNF_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T       tHead;
  PNM_AP_DCP_SET_CNF_DATA_T tData;
};


/*******************************************************************************
 DCP SET NAME SERVICE
*******************************************************************************/

/** request packet */
typedef struct PNM_AP_DCP_SET_NAME_REQ_DATA_Ttag PNM_AP_DCP_SET_NAME_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_NAME_REQ_DATA_Ttag
{
  /** MAC address of IO-device */
  uint8_t        abMacAddr[6];
  /* qualification of setting */
  uint16_t       usQualifier;
  /** new NameOfStation for IO-device */
  uint8_t        abName[240];
};

typedef struct PNM_AP_DCP_SET_NAME_REQ_Ttag PNM_AP_DCP_SET_NAME_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_NAME_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T            tHead;
  PNM_AP_DCP_SET_NAME_REQ_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_DCP_SET_NAME_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_DCP_SET_CNF_T         tCnf;
} PNM_AP_DCP_SET_NAME_PCK_T;


/*******************************************************************************
 DCP SET IP SERVICE
*******************************************************************************/

/** request packet */
typedef struct PNM_AP_DCP_SET_IP_REQ_DATA_Ttag PNM_AP_DCP_SET_IP_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_IP_REQ_DATA_Ttag
{
  /** MAC address of IO-device */
  uint8_t         abMacAddr[6];
  /* qualification of setting */
  uint16_t        usQualifier;
  /** NEW IP for IO-device */
  uint32_t        ulIpAddr;
  /** NEW subnet mask */
  uint32_t        ulNetmask;
  /** NEW router, gateway */
  uint32_t        ulGateway;
};

typedef struct PNM_AP_DCP_SET_IP_REQ_Ttag PNM_AP_DCP_SET_IP_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_IP_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T            tHead;
  PNM_AP_DCP_SET_IP_REQ_DATA_T   tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_DCP_SET_IP_REQ_T      tReq;
  /** confirmation packet */
  PNM_AP_DCP_SET_CNF_T         tCnf;
} PNM_AP_DCP_SET_IP_PCK_T;


/*******************************************************************************
 DCP SET SIGNAL SERVICE
*******************************************************************************/

#define PNM_AP_DCP_SET_SIGNAL_FREQUENCY_1_HZ   (0x0100)

/** request packet */
typedef struct PNM_AP_DCP_SET_SIGNAL_REQ_DATA_Ttag PNM_AP_DCP_SET_SIGNAL_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_SIGNAL_REQ_DATA_Ttag
{
  /** MAC address of IO-device */
  uint8_t        abMacAddr[6];
  /* qualification of setting */
  uint16_t       usQualifier;
};

typedef struct PNM_AP_DCP_SET_SIGNAL_REQ_Ttag PNM_AP_DCP_SET_SIGNAL_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_SIGNAL_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T              tHead;
  PNM_AP_DCP_SET_SIGNAL_REQ_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_DCP_SET_SIGNAL_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_DCP_SET_CNF_T           tCnf;
} PNM_AP_DCP_SET_SIGNAL_PCK_T;


/*******************************************************************************
 DCP RESET FACTORY SETTINGS SERVICE
*******************************************************************************/

/** request packet */
typedef struct PNM_AP_DCP_RESET_FACTORY_SETTINGS_REQ_DATA_Ttag PNM_AP_DCP_RESET_FACTORY_SETTINGS_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_RESET_FACTORY_SETTINGS_REQ_DATA_Ttag
{
  /** MAC address of IO-device */
  uint8_t        abMacAddr[6];
  /* qualification of setting */
  uint16_t       usQualifier;
};

typedef struct PNM_AP_DCP_RESET_FACTORY_SETTINGS_REQ_Ttag PNM_AP_DCP_RESET_FACTORY_SETTINGS_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_RESET_FACTORY_SETTINGS_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T                          tHead;
  PNM_AP_DCP_RESET_FACTORY_SETTINGS_REQ_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_DCP_RESET_FACTORY_SETTINGS_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_DCP_SET_CNF_T                       tCnf;
} PNM_AP_DCP_RESET_FACTORY_SETTINGS_PCK_T;

/*******************************************************************************
 DCP GET SERVICE
*******************************************************************************/

/* get MAC Address */
#define PNM_AP_DCP_GET_OPTION_MACADDR         (0x0001)
/* get IP Parameters */
#define PNM_AP_DCP_GET_OPTION_IP_PARAM        (0x0002)
/* get TypeOfStation */
#define PNM_AP_DCP_GET_OPTION_TYPEOFSTATION   (0x0004)
/* get NameOfStation */
#define PNM_AP_DCP_GET_OPTION_NAMEOFSTATION   (0x0008)
/* get identification (VendorId, DeviceId) */
#define PNM_AP_DCP_GET_OPTION_DEVICE_IDENT    (0x0010)
/* get device role */
#define PNM_AP_DCP_GET_OPTION_DEVICE_ROLE     (0x0020)
/* get device instance - Attention: this option is not support by devices according to Profinet specification v2.2 */
#define PNM_AP_DCP_GET_OPTION_DEVICE_INSTANCE (0x0040)

/** request packet */
typedef struct PNM_AP_DCP_GET_REQ_DATA_Ttag PNM_AP_DCP_GET_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_GET_REQ_DATA_Ttag
{
  /** MAC address of IO-device to read from */
  uint8_t        abMacAddr[6];
  /** bitfield of options to read (see above) */
  uint16_t       usRequestFlags;
};

typedef struct PNM_AP_DCP_GET_REQ_Ttag PNM_AP_DCP_GET_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_GET_REQ_Ttag
{
  /** packet header */
  PNM_AP_PCK_HEADER_T                    tHead;
  /** packet data */
  PNM_AP_DCP_GET_REQ_DATA_T              tData;
};

/** confirmation packet */
typedef struct PNM_AP_DCP_GET_CNF_DATA_Ttag PNM_AP_DCP_GET_CNF_DATA_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_GET_CNF_DATA_Ttag
{
  /** MAC address of IO-Device */
  uint8_t  abMacAddr[6];
  /** bitfield of options that returned data (see above) */
  uint16_t usValidFlags;
  /** IP address of IO-Device */
  uint32_t ulIpAddr;
  /** network mask of IO-Device */
  uint32_t ulNetmask;
  /** gateway address of IO-Device */
  uint32_t ulGateway;
  /** VendorID of IO-Device */
  uint16_t usVendorId;
  /** DeviceID of IO-Device */
  uint16_t usDeviceId;
  /** Instance of IO-Device */
  uint16_t usInstance;
  /** Role of IO-Device */
  uint16_t usRole;
  /** length of NameOfStation contained in abNameOfStation */
  uint8_t  bLenNameOfStation;
  /** length of TypeOfStation contained in abTypeOfStation */
  uint8_t  bLenTypeOfStation;
  /** NameOfStation of IO-Device */
  uint8_t  abNameOfStation[240];
  /** TypeOfStation of IO-Device */
  uint8_t  abTypeOfStation[240];
  /** amount of options that returned an error */
  uint8_t  bErrorCount;
  /** BlockError (only the last one is reported if there was more than 1 error) */
  uint8_t  bBlockError;
};

typedef struct PNM_AP_DCP_GET_CNF_Ttag PNM_AP_DCP_GET_CNF_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_GET_CNF_Ttag
{
  /** packet header */
  PNM_AP_PCK_HEADER_T                    tHead;
  /** packet data */
  PNM_AP_DCP_GET_CNF_DATA_T               tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_DCP_GET_REQ_T               tReq;
  /** confirmation packet */
  PNM_AP_DCP_GET_CNF_T               tCnf;
} PNM_AP_DCP_GET_PCK_T;


/*******************************************************************************
 GET LOGBOOK SERVICE
*******************************************************************************/

typedef struct PNM_AP_GET_LOGBOOK_REQ_DATA_Ttag PNM_AP_GET_LOGBOOK_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_GET_LOGBOOK_REQ_DATA_Ttag
{
  PNM_AP_DEVICEHANDLE_T     usDeviceHandle;

  uint16_t                  usMaxEntries;
};

typedef struct PNM_AP_GET_LOGBOOK_REQ_Ttag PNM_AP_GET_LOGBOOK_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_GET_LOGBOOK_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T           tHead;

  PNM_AP_GET_LOGBOOK_REQ_DATA_T tData;
};

enum PNM_AP_LOGBOOK_EVENT_Etag
{
  PNM_AP_LOGBOOK_EVENT_RESERVED               = 0x0000,
  PNM_AP_LOGBOOK_EVENT_USER_START             = 0x0001,

  PNM_AP_LOGBOOK_CONFIGSTATE                  = 0x0001,
  PNM_AP_LOGBOOK_CONFIGLOCKED                 = 0x0002,
  PNM_AP_LOGBOOK_CONFIGERROR                  = 0x0003,

  PNM_AP_LOGBOOK_HIF_WDGERROR                 = 0x0100,
  PNM_AP_LOGBOOK_EVENT_USER_END               = 0x7FFF,

  PNM_AP_LOGBOOK_EVENT_STACK_START            = 0x8000,

  PNM_AP_LOGBOOK_EVENT_AR_NOTESTABLISHED      = 0x8000,
  PNM_AP_LOGBOOK_EVENT_AR_ESTABLISHED         = 0x8001,
  PNM_AP_LOGBOOK_EVENT_AR_ABORT               = 0x8002,
  PNM_AP_LOGBOOK_EVENT_AR_LOW_PRIO_ALARM      = 0x8003,
  PNM_AP_LOGBOOK_EVENT_AR_HIGH_PRIO_ALARM     = 0x8004,
  PNM_AP_LOGBOOK_EVENT_AR_RPC_READ_FAILED     = 0x8005,
  PNM_AP_LOGBOOK_EVENT_AR_RPC_WRITE_FAILED    = 0x8006,
  PNM_AP_LOGBOOK_EVENT_AR_RPC_FAILED          = 0x8007,

  PNM_AP_LOGBOOK_EVENT_STACK_BUSSTATE         = 0x8100,
  PNM_AP_LOGBOOK_EVENT_STACK_MEMORYUSAGE      = 0x8101,
};

typedef enum PNM_AP_LOGBOOK_EVENT_Etag PNM_AP_LOGBOOK_EVENT_E;

typedef struct PNM_AP_LOGBOOK_RECORD_Ttag PNM_AP_LOGBOOK_RECORD_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_LOGBOOK_RECORD_Ttag
{
  /** cycle counter associated with this log record */
  uint64_t ulCycleCounter;
  /** event id of this event */
  uint16_t usEvent;
  /** reserved for future usage */
  uint16_t usPadding;
  /** additional value */
  uint32_t ulAdditionalValue;
};

/** confirmation packet */
typedef struct PNM_AP_GET_LOGBOOK_CNF_DATA_Ttag PNM_AP_GET_LOGBOOK_CNF_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_GET_LOGBOOK_CNF_DATA_Ttag
{
  PNM_AP_DEVICEHANDLE_T     usDeviceHandle;

  uint16_t                  usNumEntries;
  /** logbook entries, oldest come first */
  PNM_AP_LOGBOOK_RECORD_T   atEntries[64];
};

typedef struct PNM_AP_GET_LOGBOOK_CNF_Ttag PNM_AP_GET_LOGBOOK_CNF_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_GET_LOGBOOK_CNF_Ttag
{
  PNM_AP_PCK_HEADER_T           tHead;

  PNM_AP_GET_LOGBOOK_CNF_DATA_T tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_GET_LOGBOOK_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_GET_LOGBOOK_CNF_T    tCnf;
} PNM_AP_GET_LOGBOOK_PCK_T;


/** usable flags in ulFlags */
enum PNM_AP_SET_ARSTATUS_FLAGS_ETag
{
  PNM_AP_SET_ARSTATUS_DISABLED = PNM_AP_CFG_IOD_FLAG_DISABLED,
  /** When this bit is changed by application an SR transition is initiated:
   * - When changed from not set to set     -> AR becomes primary
   * - When changed from set     to not set -> AR becomes backup
   * The bit is handled as edge triggered! not state */
  PNM_AP_SET_ARSTATUS_SR_PRIMARY = PNM_AP_CFG_IOD_FLAG_RESERVED_SR,
};

typedef enum PNM_AP_SET_ARSTATUS_FLAGS_ETag PNM_AP_SET_ARSTATUS_FLAGS_E;

typedef struct PNM_AP_SET_ARSTATUS_REQ_DATA_Ttag PNM_AP_SET_ARSTATUS_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_SET_ARSTATUS_REQ_DATA_Ttag
{
  /** unique handle of this IO-Device defined by the sender of this packet */
  PNM_AP_DEVICEHANDLE_T usDeviceHandle;
  /* padding for alignment */
  uint16_t              usPadding;
  /** status flags see \ref PNM_AP_SET_AR_STATUS_E */
  uint32_t              ulStatus;
};

typedef struct PNM_AP_SET_ARSTATUS_REQ_Ttag PNM_AP_SET_ARSTATUS_REQ_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_SET_ARSTATUS_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T             tHead;

  PNM_AP_SET_ARSTATUS_REQ_DATA_T tData;
};

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T PNM_AP_SET_ARSTATUS_CNF_T;

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_SET_ARSTATUS_REQ_T    tReq;
  /** confirmation packet */
  PNM_AP_SET_ARSTATUS_CNF_T    tCnf;
} PNM_AP_SET_ARSTATUS_PCK_T;


enum PNM_AP_IOD_DIAGFLAG_Etag
{
  /** the device does not exist in the network */
  PNM_AP_IOD_DIAGFLAG_NOT_EXISTING       = 0x00000001,
  /** the device is not in cyclic dataexchange */
  PNM_AP_IOD_DIAGFLAG_NOT_COMMUNICATING  = 0x00000002,
  /** Multiple stations with the ip or the name exist */
  PNM_AP_IOD_DIAGFLAG_ADDRESS_CONFLICT   = 0x00000004,
  /** The device sent an faulty DCP response */
  PNM_AP_IOD_DIAGFLAG_FAULTY_DCP         = 0x00000008,
  /** The device sent an RPC response with an error */
  PNM_AP_IOD_DIAGFLAG_NEGATIVE_RPC       = 0x00000010,
  /** The device sent an RPC response with an error */
  PNM_AP_IOD_DIAGFLAG_DEACTIVATED        = 0x00000020,
  /** The device has a Module Diff Block with identification related entries */
  PNM_AP_IOD_DIAGFLAG_MODULEDIFF                  = 0x00000800,
  /** The device has a Module Diff Block with identification related entries */
  PNM_AP_IOD_DIAGFLAG_MODULEDIFF_WRONG_OR_MISSING = 0x00000800,
  /** The device has a Module Diff Block with diagnosis related entries */
  PNM_AP_IOD_DIAGFLAG_MODULEDIFF_DIAGNOSIS        = 0x00002000,
};

typedef struct PNM_AP_STORE_REMANENT_IND_DATA_Ttag PNM_AP_STORE_REMANENT_IND_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_STORE_REMANENT_IND_DATA_Ttag
{
  /** length of remanent data */
  uint16_t              usLength;
  /** Dynamic length field. usLength */
  uint8_t               abData[];
};

typedef struct PNM_AP_STORE_REMANENT_IND_Ttag PNM_AP_STORE_REMANENT_IND_T;

/** Store remanent data indication packet.
 *
 * This packet is too large to fit into the DPM mailbox, therefore
 * it will be transferred using the fragmented packet mechanism.
 */
__PACKED_PRE struct __PACKED_POST PNM_AP_STORE_REMANENT_IND_Ttag
{
  PNM_AP_PCK_HEADER_T              tHead;

  PNM_AP_STORE_REMANENT_IND_DATA_T tData;
};

/** response packet */
typedef PNM_AP_EMTPY_PCK_T PNM_AP_STORE_REMANENT_RES_T;

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_STORE_REMANENT_IND_T  tInd;
  /** confirmation packet */
  PNM_AP_STORE_REMANENT_RES_T  tRes;
} PNM_AP_STORE_REMANENT_PCK_T;

typedef struct PNM_AP_LOAD_REMANENT_REQ_DATA_Ttag PNM_AP_LOAD_REMANENT_REQ_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_LOAD_REMANENT_REQ_DATA_Ttag
{
  /** length of remanent data */
  uint16_t              usLength;
  /** Dynamic length field. usLength */
  uint8_t               abData[];
};

typedef struct PNM_AP_LOAD_REMANENT_REQ_Ttag PNM_AP_LOAD_REMANENT_REQ_T;

/** Store remanent data indication packet.
 *
 * This packet is too large to fit into the DPM mailbox, therefore
 * it will be transferred using the fragmented packet mechanism.
 */
__PACKED_PRE struct __PACKED_POST PNM_AP_LOAD_REMANENT_REQ_Ttag
{
  PNM_AP_PCK_HEADER_T             tHead;

  PNM_AP_LOAD_REMANENT_REQ_DATA_T tData;
};

/** response packet */
typedef PNM_AP_EMTPY_PCK_T PNM_AP_LOAD_REMANENT_CNF_T;

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_LOAD_REMANENT_REQ_T  tReq;
  /** confirmation packet */
  PNM_AP_LOAD_REMANENT_CNF_T  tCnf;
} PNM_AP_LOAD_REMANENT_PCK_T;


/*******************************************************************************
 DCP Set Services.

 DCP Set indication will only be reported if the application is registered and
 the Controller stack is in the right state "TODO refer to controller internal states"
*******************************************************************************/

/** response packet */
typedef PNM_AP_EMTPY_PCK_T PNM_AP_DCP_SET_RES_T;


/** DCP Set option definition */
enum PNM_AP_DCP_SET_OPTIONS_Etag
{
  PNM_AP_DCP_SET_IP       = 0x0102,
  PNM_AP_DCP_SET_NAME     = 0x0202,
  PNM_AP_DCP_SET_SIGNAL   = 0x0503,
  PNM_AP_DCP_SET_RESET    = 0x0506,
};

typedef enum PNM_AP_DCP_SET_OPTIONS_Etag  PNM_AP_DCP_SET_OPTIONS_E;

typedef struct PNM_AP_DCP_SET_SIGNAL_Ttag  PNM_AP_DCP_SET_SIGNAL_T;
/** response packet */
__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_SIGNAL_Ttag
{
  /** signal value defines duration and frequency to flash the led
   *  see \ref PNM_AP_DCP_SET_SIGNAL_FREQUENCY_1_HZ */
  uint16_t        usSignalValue;
};

typedef struct PNM_AP_DCP_SET_NAME_Ttag  PNM_AP_DCP_SET_NAME_T;
/** response packet */
__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_NAME_Ttag
{
  /* length of new name of IO-Controller */
  uint8_t        bNameLen;
  /** new NameOfStation for IO-Controller */
  uint8_t        abName[240];
};

typedef struct PNM_AP_DCP_SET_IP_Ttag  PNM_AP_DCP_SET_IP_T;
/** response packet */
__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_IP_Ttag
{
  /** NEW IP */
  uint32_t        ulIpAddr;
  /** NEW subnet mask */
  uint32_t        ulNetmask;
  /** NEW router, gateway */
  uint32_t        ulGateway;
};

typedef struct PNM_AP_DCP_SET_FACTORY_RESET_Ttag  PNM_AP_DCP_SET_FACTORY_RESET_T;
/** response packet */
__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_FACTORY_RESET_Ttag
{
  /* reset options  see \ref PNIO_APCTL_DCP_QUALIFIER_RESETTOFACTORY_APPLICATION */
  uint16_t        usResetOption;
};

typedef union PNM_AP_DCP_SET_UTag  PNM_AP_DCP_SET_U;
/** DCP set options  */
__PACKED_PRE union __PACKED_POST PNM_AP_DCP_SET_UTag
{
  PNM_AP_DCP_SET_NAME_T           tName;
  PNM_AP_DCP_SET_IP_T             tIp;
  PNM_AP_DCP_SET_SIGNAL_T         tSignal;
  PNM_AP_DCP_SET_FACTORY_RESET_T  tReset;
};

typedef struct PNM_AP_DCP_SET_IND_DATA_Ttag  PNM_AP_DCP_SET_IND_DATA_T;
/** response packet */
__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_IND_DATA_Ttag
{
  /** MAC address of IO-device */
  uint8_t           abMacAddr[6];

  /** qualification of setting  */
  uint16_t          usQualifier;

  /** DCP Option see \ref PNM_AP_DCP_SET_OPTIONS_E */
  uint16_t          usDcpOption;

  /* DCP set option */
  PNM_AP_DCP_SET_U  uDcpSet;
};

typedef struct PNM_AP_DCP_SET_IND_Ttag  PNM_AP_DCP_SET_IND_T;
__PACKED_PRE struct __PACKED_POST PNM_AP_DCP_SET_IND_Ttag
{
  PNM_AP_PCK_HEADER_T             tHead;

  PNM_AP_DCP_SET_IND_DATA_T       tData;
};

/** packet union */
typedef union
{
  /** request packet */
  PNM_AP_DCP_SET_RES_T  tRes;
  /** confirmation packet */
  PNM_AP_DCP_SET_IND_T  tInd;
} PNM_AP_DCP_SET_PCK_T;


/** AR state used in bARStatus */
enum PNM_AP_ARSTATUS_Etag
{
  PNM_AP_ARSTATUS_UNCONFIGURED                 = 0,
  PNM_AP_ARSTATUS_DISCONNECTED                 = 1,
  PNM_AP_ARSTATUS_CONNECTED                    = 2,
  PNM_AP_ARSTATUS_CONNECTED_BACKUP_NO_PRIMARY  = 3,
  PNM_AP_ARSTATUS_CONNECTED_BACKUP             = 4,
  PNM_AP_ARSTATUS_CONNECTED_PRIMARY            = 5,
};

typedef enum PNM_AP_ARSTATUS_Etag PNM_AP_ARSTATUS_E;

typedef struct PNM_AP_ARSTATUS_CHANGED_IND_DATA_Ttag PNM_AP_ARSTATUS_CHANGED_IND_DATA_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_ARSTATUS_CHANGED_IND_DATA_Ttag
{
  /** unique handle of this IO-Device defined by the sender of this packet */
  PNM_AP_DEVICEHANDLE_T usDeviceHandle;

  uint8_t               bARStatus;
};

typedef struct PNM_AP_ARSTATUS_CHANGED_IND_Ttag PNM_AP_ARSTATUS_CHANGED_IND_T;

__PACKED_PRE struct __PACKED_POST PNM_AP_ARSTATUS_CHANGED_IND_Ttag
{
  PNM_AP_PCK_HEADER_T         tHead;

  PNM_AP_ARSTATUS_CHANGED_IND_DATA_T  tData;
};

/** confirmation packet */
typedef PNM_AP_EMTPY_PCK_T PNM_AP_ARSTATUS_CHANGED_RES_T;

/** packet union */
typedef union
{
  /** indication packet */
  PNM_AP_ARSTATUS_CHANGED_IND_T       tInd;
  /** response packet */
  PNM_AP_ARSTATUS_CHANGED_RES_T       tRes;
} PNM_AP_ARSTATUS_CHANGED_PCK_T;


typedef union PNM_AP_PCK_Ttag PNM_AP_PCK_T;

union PNM_AP_PCK_Ttag
{
  PNM_AP_EMTPY_PCK_T                      tEmpty;
  PNM_AP_CFG_IOC_PCK_T                    tCfgIoc;
  PNM_AP_CFG_IOC_PRM_PCK_T                tCfgIocPrm;
  PNM_AP_CFG_IOD_PCK_T                    tCfgIod;
  PNM_AP_CFG_IOCR_PCK_T                   tCfgIocr;
  PNM_AP_CFG_SUBMODULE_PCK_T              tCfgSubModule;
  PNM_AP_CFG_AR_PRM_PCK_T                 tCfgARPrm;
  PNM_AP_CFG_RECORD_PCK_T                 tCfgRecord;
  PNM_AP_CFG_TOPO_INFO_PCK_T              tCfgTopo;
  PNM_AP_DWNL_FIN_PCK_T                   tCfgFin;
  PNM_AP_CFG_OEMPRM_PCK_T                 tCfgOemPrm;

  PNM_AP_CFG_GET_NUM_CONFIGURED_OBJECTS_PCK_T tGetCfgNumObjects;
  PNM_AP_CFG_GET_IOC_PCK_T                tGetCfgIoc;
  PNM_AP_CFG_GET_IOC_PRM_LIST_PCK_T       tGetCfgIocPrmList;
  PNM_AP_CFG_GET_IOC_PRM_PCK_T            tGetCfgIocPrm;
  PNM_AP_CFG_GET_IOD_PCK_T                tGetCfgIod;
  PNM_AP_CFG_GET_AR_PRM_LIST_PCK_T        tGetCfgArPrmList;
  PNM_AP_CFG_GET_AR_PRM_PCK_T             tGetCfgArPrm;
  PNM_AP_CFG_GET_IOCR_PCK_T               tGetCfgIocr;
  PNM_AP_CFG_GET_SUBMODULE_PCK_T          tGetCfgSubmodule;
  PNM_AP_CFG_GET_RECORD_PCK_T             tGetCfgRecord;

  PNM_AP_ESTABLISH_DAAR_PCK_T             tEstablishDAAR;
  PNM_AP_RELEASE_DAAR_PCK_T               tReleaseDAAR;

  PNM_AP_GET_ARVENDORBLOCK_RESPONSE_PCK_T tGetArVendorBlockRes;
  PNM_AP_READ_RECORD_SUBM_PCK_T           tReadRecord;
  PNM_AP_WRITE_RECORD_SUBM_PCK_T          tWriteRecord;
  PNM_AP_READ_IMPLICIT_RECORD_PCK_T       tReadImplicit;
  PNM_AP_RECV_ALARM_PCK_T                 tAlarmInd;
  PNM_AP_RECV_DIAGNOSIS_PCK_T             tDiagnosisInd;
  PNM_AP_ACK_ALARM_PCK_T                  tAckAlarm;
  PNM_AP_DCP_SET_NAME_PCK_T               tDcpSetName;
  PNM_AP_DCP_SET_IP_PCK_T                 tDcpSetIp;
  PNM_AP_DCP_SET_SIGNAL_PCK_T             tDcpSetSignal;
  PNM_AP_DCP_RESET_FACTORY_SETTINGS_PCK_T tDcpResetFactory;
  PNM_AP_DCP_GET_PCK_T                    tDcpGet;
  PNM_AP_DCP_SET_PCK_T                    tDcpSet;

  PNM_AP_STORE_REMANENT_PCK_T             tStoreRemanent;
  PNM_AP_LOAD_REMANENT_PCK_T              tLoadRemanent;

  PNM_AP_GET_LOGBOOK_PCK_T                tGetLogbook;

  PNM_AP_SET_ARSTATUS_PCK_T               tSetArStatus;
  PNM_AP_ARSTATUS_CHANGED_PCK_T           tArStatusChanged;

  /* if RCX_PUBLIC_H is included before this header, we can also make the
   * standard rcx requests we support appear in this union */
#ifdef __RCX_PUBLIC_H
  union {
    RCX_START_STOP_COMM_REQ_T    tReq;
    RCX_START_STOP_COMM_CNF_T    tCnf;
  } tRcxStartStop;
  union {
    RCX_CHANNEL_INIT_REQ_T       tReq;
    RCX_CHANNEL_INIT_CNF_T       tCnf;
  } tRcxChannelInit;
  union {
    RCX_LOCK_UNLOCK_CONFIG_REQ_T tReq;
    RCX_LOCK_UNLOCK_CONFIG_CNF_T tCnf;
  } tRcxLockUnlockConfig;
  union {
    RCX_REGISTER_APP_REQ_T       tReq;
    RCX_REGISTER_APP_CNF_T       tCnf;
  } tRcxRegisterApp;
  union {
    RCX_UNREGISTER_APP_REQ_T     tReq;
    RCX_UNREGISTER_APP_CNF_T     tCnf;
  } tRcxUnRegisterApp;
  union {
    RCX_GET_DPM_IO_INFO_REQ_T          tReq;
    RCX_GET_DPM_IO_INFO_CNF_T          tCnf;
  } tRcxGetDpmIOInfo;
  union {
    RCX_FIRMWARE_IDENTIFY_REQ_T        tReq;
    RCX_FIRMWARE_IDENTIFY_CNF_T        tCnf;
  } tRcxFirmwareIdentify;
  union {
    RCX_PACKET_GET_SLAVE_HANDLE_REQ_T  tReq;
    RCX_PACKET_GET_SLAVE_HANDLE_CNF_T  tCnf;
  } tRcxGetSlaveHandle;
  union {
    RCX_PACKET_GET_SLAVE_CONN_INFO_REQ_T tReq;
    RCX_PACKET_GET_SLAVE_CONN_INFO_CNF_T tCnf;
  } tRcxGetSlaveConnInfo;
  union {
    RCX_BUSSCAN_REQ_T  tReq;
    RCX_BUSSCAN_CNF_T  tCnf;
  } tRcxBusscan;
  union {
    RCX_GET_DEVICE_INFO_REQ_T  tReq;
    RCX_GET_DEVICE_INFO_CNF_T  tCnf;
  } tRcxGetDeviceInfo;
#endif
  /* if TLR_DIAG_H is included before this header, we can also make the
  * standard tlr diag requests we support appear in this union */
#ifdef __TLR_DIAG_H
  union {
    DIAG_INFO_GET_COMMON_STATE_REQ_T tReq;
    DIAG_INFO_GET_COMMON_STATE_CNF_T tCnf;
  } tDiagInfoGetCommonState;
  union {
    DIAG_INFO_GET_EXTENDED_STATE_REQ_T tReq;
    DIAG_INFO_GET_EXTENDED_STATE_CNF_T tCnf;
  } tDiagInfoGetExtendedState;
  union {
    DIAG_INFO_GET_WATCHDOG_TIME_REQ_T  tReq;
    DIAG_INFO_GET_WATCHDOG_TIME_CNF_T  tCnf;
  } tDiagInfoGetWatchdogTime;
  union {
    DIAG_INFO_SET_WATCHDOG_TIME_REQ_T  tReq;
    DIAG_INFO_SET_WATCHDOG_TIME_CNF_T  tCnf;
  } tDiagInfoSetWatchdogTime;
#endif
};


#if _MSC_VER >= 1000
  #pragma pack()
#endif

/*************************************************************************************/
#endif /* #ifndef __PNM_AP_PUBLIC_H */
